import ICacheDriver from '../NativeDevice/ICacheDriver';
import Property from './Property';
import CircularJSON from 'circular-json';

export interface IPropertyStorage {
	getValueOrDefault<TValue>(key: Property, defaultValue: TValue): Promise<TValue>;
	setValue<TValue>(key: string, value: TValue): Promise<void>;
	removeValue(key: string): Promise<void>;
}

const createPropertyStorage = (getNativeDriver: () => ICacheDriver): IPropertyStorage => ({
	async getValueOrDefault(key: Property, defaultValue: any) {
		const nativeDriver = getNativeDriver();
		try {
			const rawValue = await nativeDriver.cacheGet(key);
			return JSON.parse(rawValue);
		} catch (error) {
			return defaultValue;
		}
	},
	async setValue(key: string, value: any) {
		const nativeDriver = getNativeDriver();
		const rawValue = CircularJSON.stringify(value);
		await nativeDriver.cacheSave(key, rawValue);
	},
	async removeValue(key: string) {
		const nativeDriver = getNativeDriver();
		await nativeDriver.cacheDelete(key);
	},
});
export default createPropertyStorage;
