import { EventEmitter } from 'events';
import { ISerializer } from '../../Serializer/serialization';
import { ITcpSocket, SendArgs, TcpSocketEvent, TcpSocketEventListener } from './ITcpSocket';

export class MessageTcpSocket<TSerialized> implements ITcpSocket<object> {
	private emitter: EventEmitter = new EventEmitter();

	constructor(
		private socket: ITcpSocket<TSerialized>,
		private serializer: ISerializer<TSerialized>,
	) {
		this.handleSocketMessage = this.handleSocketMessage.bind(this);
		this.listenToSocket();
	}

	public async close(): Promise<void> {
		await this.socket.close();
		this.socket.removeListener(TcpSocketEvent.Message, this.handleSocketMessage);
	}

	public async send(args: SendArgs<object>): Promise<void> {
		const serializedMessage = this.serializer.serialize(args.message);
		await this.socket.send({
			host: args.host,
			port: args.port,
			message: serializedMessage,
		});
	}

	public addListener(event: TcpSocketEvent, callback: TcpSocketEventListener<object>): void {
		this.emitter.addListener(event, callback);
	}

	public removeListener(event: TcpSocketEvent, callback: TcpSocketEventListener<object>): void {
		this.emitter.removeListener(event, callback);
	}

	private listenToSocket() {
		this.socket.addListener(TcpSocketEvent.Message, this.handleSocketMessage);
	}

	private handleSocketMessage(serializedMessage: TSerialized) {
		const message = this.serializer.deserialize(serializedMessage);
		this.emitter.emit(TcpSocketEvent.Message, message);
	}
}
