import { EventEmitter } from 'events';
import Debug from 'debug';
import { locked } from '@signageos/lib/es6/Lock/lockedDecorator';
import { debug } from '@signageos/lib/es6/Debug/debugDecorator';
import wait from '@signageos/lib/dist/Timer/wait';
import { ITcpSocket, TcpSocketEvent } from './Socket/Tcp/ITcpSocket';
import {
	ClosedListener,
	IPeerNetwork,
	MessageListener,
	PeerNetworkEvent,
	PeerNetworkEventListener,
	PeerNetworkMessage,
	PeerNetworkPeer,
	PeersChangedListener,
	isPeerNetworkMessage,
} from './IPeerNetwork';
import { IPeerDiscoveryService, PeerDiscoveryServiceEvent } from './PeerDiscoveryService/IPeerDiscoveryService';

const DEBUG_NAMESPACE = '@signageos/front-display:PeerNetwork:PeerNetwork';
const logDebug = Debug(DEBUG_NAMESPACE);

export class PeerNetwork implements IPeerNetwork {
	private emitter: EventEmitter = new EventEmitter();
	private socket: ITcpSocket<object> | null = null;
	private started: boolean = false;

	constructor(
		private peerDiscoveryService: IPeerDiscoveryService,
		private createSocket: () => Promise<ITcpSocket<object>>,
	) {
		this.handlePeersChanged = this.handlePeersChanged.bind(this);
		this.handleSocketMessage = this.handleSocketMessage.bind(this);
		this.handleSocketClosed = this.handleSocketClosed.bind(this);
		this.listenToPeerChanges();
	}

	@debug(DEBUG_NAMESPACE)
	@locked('peer_network', { scope: 'instance' })
	public async start(): Promise<void> {
		if (this.started) {
			throw new Error('PeerNetwork already started');
		}

		await this.peerDiscoveryService.start();
		this.socket = await this.openSocket();
		this.started = true;
	}

	@debug(DEBUG_NAMESPACE)
	@locked('peer_network', { scope: 'instance' })
	public async stop(): Promise<void> {
		if (this.peerDiscoveryService.isStarted()) {
			await this.peerDiscoveryService.stop();
		}

		if (this.socket) {
			await this.closeSocket(this.socket);
			this.socket = null;
		}

		this.started = false;
	}

	public isStarted(): boolean {
		return this.started;
	}

	public getMe(): PeerNetworkPeer {
		const me = this.peerDiscoveryService.getMe();
		return {
			id: me.id,
			aliveAt: me.aliveAt,
		};
	}

	public getPeers(): PeerNetworkPeer[] {
		const peers = this.peerDiscoveryService.getPeers();
		return peers.map((peer) => ({
			id: peer.id,
			aliveAt: peer.aliveAt,
		}));
	}

	@debug(DEBUG_NAMESPACE)
	public async send(message: object): Promise<void> {
		if (!this.socket) {
			throw new Error('PeerNetwork is not started');
		}

		const fullMessage: PeerNetworkMessage = {
			fromId: this.getMe().id,
			message,
		};

		const peers = this.peerDiscoveryService.getPeers();

		// send to all peers
		await Promise.all(
			peers.map(async (peer) => {
				try {
					await retry({
						callback: () =>
							this.socket!.send({
								host: peer.address,
								port: peer.port,
								message: fullMessage,
							}),
						attempts: 3,
						delayBetweenAttemptsMs: 10,
					});
				} catch (error) {
					// ignore failures because we don't care if some peers are not reachable, show must go on
					console.error(`failed to send message to ${peer.address}:${peer.port}`, error);
				}
			}),
		);
	}

	public addListener(event: PeerNetworkEvent.Message, callback: MessageListener): void;
	public addListener(event: PeerNetworkEvent.PeersChanged, callback: PeersChangedListener): void;
	public addListener(event: PeerNetworkEvent.Closed, callback: ClosedListener): void;
	public addListener(event: PeerNetworkEvent, callback: PeerNetworkEventListener): void {
		this.emitter.addListener(event, callback);
	}

	public removeListener(event: PeerNetworkEvent.Message, callback: MessageListener): void;
	public removeListener(event: PeerNetworkEvent.PeersChanged, callback: PeersChangedListener): void;
	public removeListener(event: PeerNetworkEvent.Closed, callback: ClosedListener): void;
	public removeListener(event: PeerNetworkEvent, callback: PeerNetworkEventListener): void {
		this.emitter.removeListener(event, callback);
	}

	private listenToPeerChanges() {
		this.peerDiscoveryService.addListener(PeerDiscoveryServiceEvent.PeersChanged, this.handlePeersChanged);
	}

	private handlePeersChanged() {
		logDebug('peers changed');
		this.emitter.emit(PeerNetworkEvent.PeersChanged);
	}

	private async openSocket() {
		const socket = await this.createSocket();
		socket.addListener(TcpSocketEvent.Message, this.handleSocketMessage);
		socket.addListener(TcpSocketEvent.Closed, this.handleSocketClosed);
		return socket;
	}

	private async closeSocket(socket: ITcpSocket<object>) {
		socket.removeListener(TcpSocketEvent.Message, this.handleSocketMessage);
		socket.removeListener(TcpSocketEvent.Closed, this.handleSocketClosed);
		await socket.close();
	}

	private handleSocketMessage(message: object) {
		if (isPeerNetworkMessage(message)) {
			this.emitter.emit(PeerNetworkEvent.Message, message);
		}
	}

	private async handleSocketClosed(error?: Error) {
		logDebug('socket closed', error);
		await this.stop();
		this.emitter.emit(PeerNetworkEvent.Closed, error);
	}
}

async function retry({
	callback,
	attempts,
	delayBetweenAttemptsMs,
}: {
	callback: () => Promise<void>;
	attempts: number;
	delayBetweenAttemptsMs: number;
}) {
	let lastError: Error | undefined;
	for (let i = 0; i < attempts; i++) {
		try {
			await callback();
			return;
		} catch (error) {
			lastError = error;
			await wait(delayBetweenAttemptsMs);
		}
	}

	throw lastError;
}
