import { EventEmitter } from 'events';
import Debug from 'debug';
import { debug } from '@signageos/lib/dist/Debug/debugDecorator';
import { IPeerDiscoveryService, PeerDiscoveryPeer, PeerDiscoveryServiceEvent, PeersChangedCallback } from './IPeerDiscoveryService';
import { PeerDiscoveryServiceMessageSender } from './PeerDiscoveryServiceMessageHandler';
import { IUdpSocket } from '../Socket/Udp/IUdpSocket';
import { NetworkInfo } from './INetworkInfo';

const DEBUG_NAMESPACE = '@signageos/front-display:PeerNetwork:PeerDiscoveryService';
const logDebug = Debug(DEBUG_NAMESPACE);

export interface PeerDiscoveryServiceConfig {
	pingIntervalMs: number;
	cleanDeadPeersIntervalMs: number;
	aliveTimeout: number;
}

const defaultConfig = {
	pingIntervalMs: 5e3,
	cleanDeadPeersIntervalMs: 30e3,
	aliveTimeout: 30e3,
};

/**
 * Handles discovery of other peers in the network and communication with them.
 */
export class PeerDiscoveryService implements IPeerDiscoveryService {
	private config: PeerDiscoveryServiceConfig;

	private peers: { [id: string]: PeerDiscoveryPeer } = {};

	private emitter: EventEmitter = new EventEmitter();
	private messageHandler: PeerDiscoveryServiceMessageSender;

	private pingInterval: ReturnType<typeof setInterval> | null = null;
	private cleanInterval: ReturnType<typeof setInterval> | null = null;

	constructor(
		private id: string,
		private socketFactory: () => Promise<IUdpSocket<object>>,
		private getNetworkInfo: () => Promise<NetworkInfo | null>,
		config: Partial<PeerDiscoveryServiceConfig> = {},
	) {
		logDebug('initiated with id ' + this.id);
		this.config = { ...defaultConfig, ...config };
		this.messageHandler = new PeerDiscoveryServiceMessageSender(this.id, this.socketFactory);
		this.registerMessageHandlerListeners();
	}

	@debug(DEBUG_NAMESPACE)
	public async start(): Promise<void> {
		if (this.messageHandler.isStarted()) {
			throw new Error('PeerDiscoveryService already started');
		}

		const networkInfo = await this.getNetworkInfo();
		if (!networkInfo) {
			throw new Error('Cannot obtain network info');
		}

		await this.messageHandler.start();

		try {
			await this.messageHandler.sendQuery();
			await this.messageHandler.sendAnnounce(networkInfo.address, networkInfo.port);
		} catch (error) {
			console.error('PeerDiscoveryService: Failed to send initial messages query and announce', error);
		}

		this.pingInterval = setInterval(() => this.messageHandler.sendPing(), this.config.pingIntervalMs);
		this.cleanInterval = setInterval(() => this.cleanDeadPeers(), this.config.cleanDeadPeersIntervalMs);
	}

	@debug(DEBUG_NAMESPACE)
	public async stop(): Promise<void> {
		if (!this.messageHandler.isStarted()) {
			throw new Error('PeerDiscoveryService already stopped');
		}

		await this.messageHandler.sendRenounce();

		if (this.pingInterval) {
			clearInterval(this.pingInterval);
			this.pingInterval = null;
		}
		if (this.cleanInterval) {
			clearInterval(this.cleanInterval);
			this.cleanInterval = null;
		}

		await this.messageHandler.stop();
	}

	public isStarted(): boolean {
		return this.messageHandler.isStarted();
	}

	public getMe() {
		return {
			id: this.id,
			aliveAt: new Date(),
		};
	}

	public getPeers(): PeerDiscoveryPeer[] {
		return Object.keys(this.peers).map((key) => this.peers[key]);
	}

	public addListener(event: PeerDiscoveryServiceEvent.PeersChanged, callback: PeersChangedCallback) {
		this.emitter.addListener(event, callback);
	}

	public removeListener(event: PeerDiscoveryServiceEvent.PeersChanged, callback: PeersChangedCallback) {
		this.emitter.removeListener(event, callback);
	}

	private registerMessageHandlerListeners() {
		this.messageHandler.onQuery((message) => this.handleQueryMessage(message.sourceId));
		this.messageHandler.onAnnounce((message) =>
			this.handleAnnounceMessage({
				sourceId: message.sourceId,
				address: message.address,
				port: message.port,
			}),
		);
		this.messageHandler.onRenounce((message) => this.handleRenounce(message.sourceId));
		this.messageHandler.onPing((message) => this.handlePingMessage(message.sourceId));
	}

	private async handleQueryMessage(sourceId: string) {
		logDebug('received query message from ' + sourceId);

		const networkInfo = await this.getNetworkInfo();
		if (!networkInfo) {
			logDebug('Cannot obtain network info. Query message will be discarded without reply.');
			return;
		}

		await this.messageHandler.sendAnnounce(networkInfo.address, networkInfo.port);
	}

	private handleAnnounceMessage({ sourceId, address, port }: { sourceId: string; address: string; port: number }) {
		logDebug('received announce message from ' + sourceId, { address, port });

		const peer: PeerDiscoveryPeer = {
			id: sourceId,
			aliveAt: new Date(),
			address,
			port,
		};

		const oldPeer = this.peers[sourceId];

		if (!oldPeer || oldPeer.address !== address || oldPeer.port !== port) {
			if (oldPeer) {
				logDebug(`Peer ${oldPeer.id} changed`, peer);
			} else {
				logDebug(`New peer ${peer.id}`, peer);
			}
			this.peers[sourceId] = peer;
			logDebug('Peers changed', this.peers);
			this.emitter.emit(PeerDiscoveryServiceEvent.PeersChanged);
		}
	}

	private handleRenounce(sourceId: string) {
		logDebug('received renounce message from ' + sourceId);

		if (this.peers[sourceId] !== undefined) {
			logDebug(`Peer ${sourceId} renounced`);
			delete this.peers[sourceId];
			logDebug('Peers changed', this.peers);
			this.emitter.emit(PeerDiscoveryServiceEvent.PeersChanged);
		}
	}

	private async handlePingMessage(sourceId: string) {
		logDebug('received ping message from ' + sourceId);

		if (this.peers[sourceId] !== undefined) {
			this.peers[sourceId].aliveAt = new Date();
		} else {
			await this.messageHandler.sendQuery();
		}
	}

	private cleanDeadPeers() {
		for (const peerId in this.peers) {
			const aliveAt = this.peers[peerId].aliveAt;
			const deadAt = aliveAt.valueOf() + this.config.aliveTimeout;
			const now = Date.now();

			if (deadAt <= now) {
				logDebug(`${peerId} is dead, renouncing`);
				this.handleRenounce(peerId);
			}
		}
	}
}
