import AsyncLock from 'async-lock';
import { posix as path } from 'path';
import _ from 'lodash';
import { wait } from '@signageos/lib/dist/DateTime/timer';
import { IStorageUnit, IFilePath } from '../NativeDevice/fileSystem';
import IFileSystem from '../NativeDevice/IFileSystem';
import HashAlgorithm from '../NativeDevice/HashAlgorithm';
import FileNotFoundError from '../Front/Applet/Error/FileNotFoundError';
import ErrorCodes from '../Front/Applet/Error/ErrorCodes';
import ErrorSuggestions from '../Front/Applet/Error/ErrorSuggestions';

const DATA_PATH_PREFIX = 'data';

export default class OfflineCache {
	private asyncLock: AsyncLock;
	private storageUnitsCache: IStorageUnit[] | null = null;

	constructor(private fileSystem: IFileSystem) {
		this.asyncLock = new AsyncLock();
		this.fileSystem.onStorageUnitsChanged(() => (this.storageUnitsCache = null));
		setInterval(() => (this.storageUnitsCache = null), 30e3);
	}

	public async retriableDownloadFile(
		countOfRetrials: number,
		uid: string,
		uri: string,
		headers?: { [key: string]: string },
		forceInternalStorageUnit: boolean = false,
	) {
		try {
			const storageUnit = await this.getMostFreeStorageUnit(forceInternalStorageUnit);
			const targetFilePath = { storageUnit, filePath: this.addDataPrefixToUid(uid) };
			await this.ensureDirectory({ storageUnit, filePath: this.getParentDirectoryPath(targetFilePath.filePath) });
			await this.fileSystem.downloadFile(targetFilePath, uri, headers);
		} catch (error) {
			console.info('Failed to download ' + uid + '. Retrying ' + countOfRetrials, error);
			if (countOfRetrials > 0) {
				const defferSeconds = Math.pow(2, Math.max(5 - countOfRetrials, 0));
				await wait(window, defferSeconds * 1e3);
				await this.retriableDownloadFile(countOfRetrials - 1, uid, uri, headers);
			} else {
				throw error;
			}
		}
	}

	public async fileExists(uid: string) {
		const prefixedPath = this.addDataPrefixToUid(uid);
		const storageUnits = await this.getStorageUnits();
		for (const storageUnit of storageUnits) {
			if (await this.fileSystem.exists({ storageUnit, filePath: prefixedPath })) {
				return true;
			}
		}

		return false;
	}

	public async fileIsDirectory(uid: string) {
		const prefixedPath = this.addDataPrefixToUid(uid);
		const storageUnits = await this.getStorageUnits();
		for (const storageUnit of storageUnits) {
			if (await this.fileSystem.exists({ storageUnit, filePath: prefixedPath })) {
				return await this.fileSystem.isDirectory({ storageUnit, filePath: prefixedPath });
			}
		}

		throw this.createFileNotFoundError(uid);
	}

	public async getFile(uid: string) {
		const prefixedPath = this.addDataPrefixToUid(uid);
		const storageUnits = await this.getStorageUnits();
		for (const storageUnit of storageUnits) {
			const filePath = { storageUnit, filePath: prefixedPath };
			if (await this.fileSystem.exists(filePath)) {
				return await this.fileSystem.getFile(filePath);
			}
		}

		throw this.createFileNotFoundError(uid);
	}

	public async getFullFilePath(uid: string): Promise<IFilePath> {
		const prefixedPath = this.addDataPrefixToUid(uid);
		const storageUnits = await this.getStorageUnits();
		for (const storageUnit of storageUnits) {
			const filePath = { storageUnit, filePath: prefixedPath };
			if (await this.fileSystem.exists(filePath)) {
				return filePath;
			}
		}

		throw this.createFileNotFoundError(uid);
	}

	public async readFile(uid: string) {
		const prefixedPath = this.addDataPrefixToUid(uid);
		const storageUnits = await this.getStorageUnits();
		for (const storageUnit of storageUnits) {
			const filePath = { storageUnit, filePath: prefixedPath };
			if (await this.fileSystem.exists(filePath)) {
				return await this.fileSystem.readFile(filePath);
			}
		}

		throw this.createFileNotFoundError(uid);
	}

	public async deleteFile(uid: string, recursive: boolean) {
		if (!(await this.fileExists(uid))) {
			throw this.createFileNotFoundError(uid);
		}

		const prefixedPath = this.addDataPrefixToUid(uid);
		const storageUnits = await this.getStorageUnits();
		for (const storageUnit of storageUnits) {
			const filePath = { storageUnit, filePath: prefixedPath };
			if (await this.fileSystem.exists(filePath)) {
				await this.fileSystem.deleteFile(filePath, recursive);
			}
		}
	}

	public async deleteFileAndDeleteDirectoryIfEmpty(uid: string, recursive: boolean = false) {
		await this.deleteFile(uid, recursive);
		const parentDirectoryUid = this.getParentDirectoryPath(uid);
		if (parentDirectoryUid !== '') {
			// do not delete root directory
			await this.deleteDirectoryIfEmpty(parentDirectoryUid);
		}
	}

	public async getFileChecksum(uid: string, hashType: HashAlgorithm) {
		const prefixedPath = this.addDataPrefixToUid(uid);
		const storageUnits = await this.getStorageUnits();
		for (const storageUnit of storageUnits) {
			const filePath = { storageUnit, filePath: prefixedPath };
			if (await this.fileSystem.exists(filePath)) {
				return await this.fileSystem.getFileChecksum(filePath, hashType);
			}
		}

		throw this.createFileNotFoundError(uid);
	}

	public async extractFile(archiveUid: string, destinationDirectoryUid: string, method: string, forceInternalStorageUnit: boolean = false) {
		const prefixedArchivePath = this.addDataPrefixToUid(archiveUid);
		const prefixedDestinationDirectoryPath = this.addDataPrefixToUid(destinationDirectoryUid);
		const storageUnits = await this.getStorageUnits();
		for (const storageUnit of storageUnits) {
			const archiveFilePath = { storageUnit, filePath: prefixedArchivePath };
			if (await this.fileSystem.exists(archiveFilePath)) {
				const destinationStorageUnit = await this.getMostFreeStorageUnit(forceInternalStorageUnit);
				const destinationDirectoryPath = { storageUnit: destinationStorageUnit, filePath: prefixedDestinationDirectoryPath };
				const destinationFileUids = await this.listFilesRecursively(destinationDirectoryUid);
				if (await this.fileExists(destinationDirectoryUid)) {
					await Promise.all(
						destinationFileUids.map((destinationFileUid: string) => this.deleteFileAndDeleteDirectoryIfEmpty(destinationFileUid)),
					);
				}
				await this.ensureDirectory(destinationDirectoryPath);
				return await this.fileSystem.extractFile(archiveFilePath, destinationDirectoryPath, method);
			}
		}

		throw new FileNotFoundError({
			kind: 'fileNotFoundError',
			message: `Archive file ${archiveUid} was not found`,
			code: ErrorCodes.FILE_NOT_FOUND,
			suggestion: ErrorSuggestions.FILE_NOT_FOUND,
		});
	}

	public async getArchiveInfo(uid: string) {
		const prefixedPath = this.addDataPrefixToUid(uid);
		const storageUnits = await this.getStorageUnits();
		for (const storageUnit of storageUnits) {
			const filePath = { storageUnit, filePath: prefixedPath };
			if (await this.fileSystem.exists(filePath)) {
				return await this.fileSystem.getArchiveInfo(filePath);
			}
		}
		throw this.createFileNotFoundError(uid);
	}

	public async listFilesRecursively(directoryUid: string = ''): Promise<string[]> {
		if (!(await this.fileExists(directoryUid))) {
			return [];
		}
		const fileUids = await this.listFileUids(directoryUid);
		const recursiveFileUids = fileUids.map(async (fileUid: string) => {
			if (await this.fileIsDirectory(fileUid)) {
				return await this.listFilesRecursively(fileUid);
			} else {
				return fileUid;
			}
		});
		return _.flatten(await Promise.all(recursiveFileUids));
	}

	public async listFileUids(directoryUid: string): Promise<string[]> {
		const prefixedDirectoryPath = this.addDataPrefixToUid(directoryUid);
		const storageUnits = await this.getStorageUnits();
		return _.uniq(
			_.flatten(
				await Promise.all(
					storageUnits.map(async (storageUnit: IStorageUnit) => {
						const directoryPath = { storageUnit, filePath: prefixedDirectoryPath };
						if (await this.fileSystem.exists(directoryPath)) {
							const filePaths = await this.fileSystem.listFiles(directoryPath);
							return filePaths.map((filePath: IFilePath) => this.stripDataPrefixFromPath(filePath.filePath));
						} else {
							return [];
						}
					}),
				),
			),
		);
	}

	private async deleteDirectoryIfEmpty(uid: string) {
		await this.asyncLock.acquire(`cleanupDirectory.${uid}`, async () => {
			if (await this.fileExists(uid)) {
				const files = await this.listFileUids(uid);
				if (files.length === 0) {
					await this.deleteFileAndDeleteDirectoryIfEmpty(uid);
				}
			}
		});
	}

	private async ensureDirectory(directoryPath: IFilePath) {
		if (!(await this.fileSystem.exists(directoryPath))) {
			if (!this.isRootFilePath(directoryPath)) {
				await this.ensureDirectory({
					storageUnit: directoryPath.storageUnit,
					filePath: this.getParentDirectoryPath(directoryPath.filePath),
				});
			}
			await this.fileSystem.createDirectory(directoryPath);
		}
	}

	private getParentDirectoryPath(directoryPath: string): string {
		const parentDirectoryPath = path.dirname(directoryPath);
		if (parentDirectoryPath === '.') {
			return '';
		} else {
			return parentDirectoryPath;
		}
	}

	private isRootFilePath(filePath: IFilePath) {
		return filePath.filePath === DATA_PATH_PREFIX;
	}

	private async getMostFreeStorageUnit(onlyInternal: boolean = false) {
		let storageUnits = await this.getStorageUnits();
		if (onlyInternal) {
			storageUnits = storageUnits.filter((storageUnit: IStorageUnit) => !storageUnit.removable);
		}
		return storageUnits.sort((a: IStorageUnit, b: IStorageUnit) => b.usableSpace - a.usableSpace)[0];
	}

	private async getStorageUnits() {
		if (this.storageUnitsCache !== null) {
			return this.storageUnitsCache;
		}
		const storageUnits = await this.fileSystem.listStorageUnits();
		this.storageUnitsCache = storageUnits;
		return storageUnits;
	}

	private addDataPrefixToUid(filePath: string) {
		if (filePath === '') {
			return DATA_PATH_PREFIX;
		}
		return DATA_PATH_PREFIX + '/' + filePath;
	}

	private stripDataPrefixFromPath(filePath: string) {
		let uid = filePath;
		if (uid.startsWith(DATA_PATH_PREFIX)) {
			uid = uid.slice(DATA_PATH_PREFIX.length);
		}
		if (uid.startsWith('/')) {
			return uid.slice(1);
		}
		return uid;
	}

	private createFileNotFoundError(fileUid: string): FileNotFoundError {
		return new FileNotFoundError({
			kind: 'fileNotFoundError',
			message: `File ${fileUid} was not found`,
			code: ErrorCodes.FILE_NOT_FOUND,
			suggestion: ErrorSuggestions.FILE_NOT_FOUND,
		});
	}
}
