import { ActiveAppletDispatchCommand } from '@signageos/actions/dist/Applet/activeAppletActions';
import { SaveOfflineActionsFile } from '@signageos/actions/dist/Offline/offlineActions';
import { fork, takeEvery } from 'redux-saga/effects';
import { IFrontManagementDriver } from '../NativeDevice/Management/IManagementDriver';
import { SaveOfflineAction } from './offlineActions';
import { runPeriodicTaskSagaWhilePlatform } from '../Saga/periodicTaskSaga';
import { NO_FILE, OfflineActionsUploader } from './OfflineActionsUploader';
import IBasicDriver from '../NativeDevice/IBasicDriver';
import { awaitDependencies, withDependencies } from '../DI/dependencyInjection';
import { deliver } from '../Socket/socketActionDeliverHelper';
import { SystemLog } from '@signageos/actions/dist/SystemLog/systemLogActions';
import { ThrottledActionsChecker } from '../Socket/ThrottledActionsChecker';

const ACTIONS_TO_SAVE = [ActiveAppletDispatchCommand, SystemLog];

export const offlineActionFile = 'offlineActions';

interface OfflineActionsSagaOptions {
	initDelayMs?: number;
	periodMs?: number;
}

export function* offlineActionsSaga(
	fileNamePrefix: string,
	getNativeDriver: () => IBasicDriver,
	getManagementDriver: () => IFrontManagementDriver,
	options?: OfflineActionsSagaOptions,
) {
	const uploader = new OfflineActionsUploader(fileNamePrefix, getManagementDriver().fileSystem);
	const { offlineActionsLimits } = yield awaitDependencies(['offlineActionsLimits']);
	const actionChecker = new ThrottledActionsChecker(uploader, offlineActionsLimits);
	const deviceUid: string = yield getNativeDriver().getDeviceUid();
	yield fork(
		withDependencies(['uploadBaseUrl'], function* ({ uploadBaseUrl }) {
			yield runPeriodicTaskSagaWhilePlatform(
				options?.initDelayMs ?? 60e3,
				options?.periodMs ?? 60e3,
				function* (): IterableIterator<any> {
					yield uploadOfflineActions(uploader, offlineActionFile, deviceUid, uploadBaseUrl);
				},
				{ randomize: true },
			);
		}),
	);

	yield takeEvery(SaveOfflineAction, function* (action: SaveOfflineAction<any>): IterableIterator<any> {
		try {
			if (ACTIONS_TO_SAVE.includes(action.action.type)) {
				yield actionChecker.addActionOrPass(action.action);
			}
		} catch (e) {
			console.error(`offline action ${action.action.type} failed`, e);
		}
	});
}

function* uploadOfflineActions(
	uploader: OfflineActionsUploader,
	fileName: string,
	deviceUid: string,
	uploadBaseUrl: string,
): IterableIterator<any> {
	try {
		const response = yield uploader.upload(deviceUid, fileName, uploadBaseUrl);
		if (response !== null) {
			if (response !== NO_FILE) {
				try {
					const { uri } = JSON.parse(response!);
					yield deliver<SaveOfflineActionsFile>({
						type: SaveOfflineActionsFile,
						actionType: SystemLog,
						fileUri: uri,
					});
				} catch (e) {
					console.error(`upload offline actions file ${fileName} failed: expected JSON response`, e);
				}
			}
		} else {
			console.error(`upload offline actions file ${fileName} failed: didn\'t get a response`);
		}
	} catch (e) {
		console.error(`upload offline actions file ${fileName} failed`, e);
	}
}
