import _ from 'lodash';
import path from 'path';
import IOfflineAction from '@signageos/actions/dist/IOfflineAction';
import IFileSystem from '../NativeDevice/IFileSystem';
import { IFilePath, IStorageUnit } from '../NativeDevice/fileSystem';
import { locked } from '../Lock/lockedDecorator';
import { debug } from '@signageos/lib/dist/Debug/debugDecorator';
import IAction from '@signageos/actions/dist/IAction';
import { offlineActionFile } from './offlineActionsSagas';
import { SystemLog } from '@signageos/actions/dist/SystemLog/systemLogActions';

const OFFLINE_DIR = '.offline-actions';
const LOCK_KEY = 'OfflineActionsUploader';
const DEBUG_NAMESPACE = '@signageos/front-display:Offline:OfflineActionsUploader';
const REFRESH_SPACE_INTERVAL = 60e3; // 1 minute
const MINIMAL_STORAGE_FREE_SPACE = 50 * 1024;
const FRONT_FILE = 'front';
const MANAGEMENT_FILE = 'management';
export const NO_FILE = 'no_file_for_upload';
export class OfflineActionsUploader {
	private deviceFreeSpace: number | undefined;
	private storageInterval: NodeJS.Timeout;

	constructor(
		private fileNamePrefix: string,
		private fileSystem: IFileSystem,
	) {}

	@locked(LOCK_KEY)
	@debug(DEBUG_NAMESPACE)
	public async addAction(action: IAction<string> & IOfflineAction) {
		this.start();
		if (this.deviceFreeSpace && this.deviceFreeSpace < MINIMAL_STORAGE_FREE_SPACE) {
			console.warn('No space for saving offline actions');
		} else {
			await this.ensureOfflineDir();
			const appletCommandsFilePath = await this.getFilePath(offlineActionFile, this.getFileNumber());
			await this.fileSystem.appendFile(appletCommandsFilePath, this.actionToFileEntry(action));
		}
	}

	@locked(LOCK_KEY)
	@debug(DEBUG_NAMESPACE)
	public async upload(deviceUid: string, fileName: string, baseUrl: string) {
		const filePath = await this.getFilePathUpload(fileName);
		if (await this.fileSystem.exists(filePath)) {
			const uploadUri = this.getUploadUri(baseUrl, deviceUid);
			const response = await this.fileSystem.uploadFile(filePath, uploadUri, 'file');
			if (response?.includes(SystemLog)) {
				await this.fileSystem.deleteFile(filePath, false);
				return response;
			} else {
				return response;
			}
		} else {
			this.stop();
			return NO_FILE;
		}
	}

	private async ensureOfflineDir() {
		const offlineDirPath = await this.getOfflineDirFilePath();
		await this.ensureDir(offlineDirPath);
	}

	private async ensureDir(dirPath: IFilePath) {
		if (await this.fileSystem.exists(dirPath)) {
			if (!(await this.fileSystem.isDirectory(dirPath))) {
				await this.fileSystem.deleteFile(dirPath, true);
				await this.fileSystem.createDirectory(dirPath);
			}
		} else {
			await this.fileSystem.createDirectory(dirPath);
		}
	}

	private async getFilePath(fileName: string, fileNumber: number): Promise<IFilePath> {
		const dirPath = await this.getOfflineDirFilePath();
		return {
			...dirPath,
			filePath: path.join(dirPath.filePath, this.fileNamePrefix + fileName + '.' + fileNumber),
		};
	}

	private async getOfflineDirFilePath(): Promise<IFilePath> {
		const storageUnits = await this.fileSystem.listStorageUnits();
		const internalStorageUnit = storageUnits.filter((storageUnit: IStorageUnit) => !storageUnit.removable)[0];
		return {
			filePath: OFFLINE_DIR,
			storageUnit: internalStorageUnit,
		};
	}

	private actionToFileEntry(action: IAction<string> & IOfflineAction) {
		const sanitizedAction = _.omit(action, '__offline');
		return JSON.stringify(sanitizedAction) + '\n';
	}

	private getUploadUri(baseUrl: string, deviceUid: string) {
		const UPLOAD_PATH = '/upload/file';
		const prefix = `offline-actions/${SystemLog}/${deviceUid}/`;
		return `${baseUrl}${UPLOAD_PATH}?prefix=${prefix}`;
	}

	private async updateDeviceSpace() {
		const localStorages = await this.fileSystem.listStorageUnits();
		const internalStorage = localStorages.find((storage) => !storage.removable);
		if (internalStorage) {
			this.deviceFreeSpace = internalStorage?.freeSpace;
		}
	}

	private getFileNumber() {
		const dates = new Date();
		return dates.getFullYear() + dates.getMonth() + dates.getDay() + dates.getHours();
	}
	private async getFilePathUpload(fileName: string): Promise<IFilePath> {
		const dirPath = await this.getOfflineDirFilePath();
		let filePath: IFilePath;
		const directoryExists = await this.fileSystem.exists(dirPath);
		if (!directoryExists) {
			filePath = {
				...dirPath,
				filePath: path.join(dirPath.filePath, this.fileNamePrefix + fileName),
			};
		} else {
			const dirContent: IFilePath[] = await this.fileSystem.listFiles(dirPath);

			switch (dirContent.length) {
				case 0:
					filePath = {
						...dirPath,
						filePath: path.join(dirPath.filePath, this.fileNamePrefix + fileName),
					};
					break;
				case 1:
					const rightFilePath = this.checkFilePath(dirContent[0].filePath);
					filePath = {
						...dirPath,
						filePath: rightFilePath,
					};
					break;
				default:
					let files = this.fileNamePrefix.includes(FRONT_FILE)
						? dirContent.filter((file) => file.filePath.includes(FRONT_FILE))
						: dirContent.filter((file) => file.filePath.includes(MANAGEMENT_FILE));
					const rightFilePathFiles = this.checkFilePath(files[0].filePath);
					filePath = {
						...dirPath,
						filePath: rightFilePathFiles,
					};
					break;
			}
		}
		return filePath;
	}

	private checkFilePath(filePath: string) {
		return !filePath.startsWith('.') ? '.' + filePath : filePath;
	}

	private start() {
		const deviceFreeSpace = this.deviceFreeSpace;
		if (deviceFreeSpace === undefined) {
			this.storageInterval = setInterval(() => {
				this.updateDeviceSpace();
			}, REFRESH_SPACE_INTERVAL);
		}
	}

	private stop() {
		if (this.storageInterval) {
			this.deviceFreeSpace = undefined;
			clearInterval(this.storageInterval);
		}
	}
}
