import { overrideDateCurrentTimestamp } from '@signageos/lib/dist/DateTime/dateMonkeyPatch';
import * as momentTimezone from 'moment-timezone';
import Property from '../../Property/Property';
import ITimeManager from './ITimeManager';
import ZoneOffset from './ZoneOffset';

type OverridenTimestampSettings = {
	timestamp: number;
	updatedAt: number;
};

export default class EmulatorTimeManager implements ITimeManager {
	private static OVERRIDEN_TIMESTAMP_SETTINGS_KEY: string = 'default.native_device.OVERRIDEN_TIMESTAMP';
	private static OVERRIDEN_TIMEZONE_OFFSET_KEY: string = Property.CURRENT_TIMEZONE_OFFSET;
	private static OVERRIDEN_TIMEZONE_KEY: string = 'default.native_device.OVERRIDEN_TIMEZONE';
	private originalDate: DateConstructor;

	constructor(private window: Window) {
		this.originalDate = overrideDateCurrentTimestamp(window as any, () => this.getOverridenTimestamp());
	}

	public async getEpochMillis() {
		return this.getOverridenTimestamp();
	}

	public async getZoneOffset() {
		const overridenTimezoneOffset = this.loadOverridenTimezoneOffset();
		const originalTimezoneOffset = new this.originalDate().getTimezoneOffset();
		return new ZoneOffset(overridenTimezoneOffset ?? originalTimezoneOffset);
	}

	public async canGetZoneId() {
		return true;
	}

	public async getZoneId() {
		return this.loadOverridenTimezone() ?? Intl.DateTimeFormat().resolvedOptions().timeZone;
	}

	public async canGetNtpServer() {
		return false;
	}

	public async getNtpServer(): Promise<string | null> {
		throw new Error('Device is unable to return NTP server');
	}

	public async canSetManual() {
		return true;
	}

	public async setManual(epochMillis: number, zoneId: string) {
		const timezoneOffset = momentTimezone.tz(epochMillis, zoneId).utcOffset();
		this.saveOverringTimezoneOffset(timezoneOffset);
		this.saveOverridenTimezone(zoneId);
		this.saveOverridenTimestampSettings({ timestamp: epochMillis, updatedAt: this.originalDate.now() });
	}

	public async canSetAuto() {
		return false;
	}

	public async setAuto() {
		throw new Error('NTP server not supported');
	}

	public async isAutoEnabled() {
		return false;
	}

	private getOverridenTimestamp() {
		const overridenTimestampSettings = this.loadOverridenTimestampSettings();
		if (overridenTimestampSettings) {
			return this.originalDate.now() - overridenTimestampSettings.updatedAt + overridenTimestampSettings.timestamp;
		}
		return this.originalDate.now();
	}

	private loadOverridenTimestampSettings() {
		const overridenTimestamp = this.window.localStorage.getItem(EmulatorTimeManager.OVERRIDEN_TIMESTAMP_SETTINGS_KEY);
		if (overridenTimestamp) {
			return JSON.parse(overridenTimestamp) as OverridenTimestampSettings;
		}
		return null;
	}

	private saveOverridenTimestampSettings(settings: OverridenTimestampSettings) {
		this.window.localStorage.setItem(EmulatorTimeManager.OVERRIDEN_TIMESTAMP_SETTINGS_KEY, JSON.stringify(settings));
	}

	private loadOverridenTimezoneOffset() {
		return Number(this.window.localStorage.getItem(EmulatorTimeManager.OVERRIDEN_TIMEZONE_OFFSET_KEY));
	}

	private saveOverringTimezoneOffset(timezoneOffset: number) {
		this.window.localStorage.setItem(EmulatorTimeManager.OVERRIDEN_TIMEZONE_OFFSET_KEY, String(timezoneOffset));
	}

	private loadOverridenTimezone() {
		return this.window.localStorage.getItem(EmulatorTimeManager.OVERRIDEN_TIMEZONE_KEY);
	}

	private saveOverridenTimezone(timezone: string) {
		this.window.localStorage.setItem(EmulatorTimeManager.OVERRIDEN_TIMEZONE_KEY, timezone);
	}
}
