import IVideoPlayer from '../../Video/IVideoPlayer';
import IStreamPlayer from '../../Stream/IStreamPlayer';
import IScreenRotationManager from './IScreenRotationManager';
import CSSRotationVideoPlayerAdapter from './CSSRotationVideoPlayerAdapter';
import CSSRotationStreamPlayerAdapter from './CSSRotationStreamPlayerAdapter';
import IScreenRotationPropsProvider from './IScreenRotationPropsProvider';
import { IHTMLElementProducer } from '../../HTML/IHTMLElementProducer';
import Orientation from '@signageos/common-types/dist/Device/Resolution/Orientation';

export default class ScreenRotationManager implements IScreenRotationManager {
	constructor(
		private window: Window,
		private propsProvider: IScreenRotationPropsProvider,
		private elements: HTMLElement[],
		private getOrientation: () => Promise<Orientation>,
	) {}

	public async applyOrientation() {
		await this.applyOrientationToElements(this.elements);
	}

	public adaptVideoPlayer(videoPlayer: IVideoPlayer): IVideoPlayer {
		return new CSSRotationVideoPlayerAdapter(this.window, videoPlayer, async () => {
			const orientation = await this.getOrientation();
			const rotationProps = this.propsProvider.getProps(orientation);
			return rotationProps.angle;
		});
	}

	public adaptStreamPlayer(streamPlayer: IStreamPlayer): IStreamPlayer {
		return new CSSRotationStreamPlayerAdapter(this.window, streamPlayer, async () => {
			const orientation = await this.getOrientation();
			const rotationProps = this.propsProvider.getProps(orientation);
			return rotationProps.angle;
		});
	}

	public adaptHTMLElementProducer(htmlElementProducer: IHTMLElementProducer): IHTMLElementProducer {
		htmlElementProducer.onElementCreated(async (element: HTMLElement) => {
			await this.applyOrientationToElements([element]);
		});
		return htmlElementProducer;
	}

	private async applyOrientationToElements(elements: HTMLElement[]) {
		const orientation = await this.getOrientation();
		const rotationProps = this.propsProvider.getProps(orientation);
		for (const element of elements) {
			element.style.width = rotationProps.width;
			element.style.height = rotationProps.height;
			element.style.transform = element.style.webkitTransform =
				`rotate(${rotationProps.angle}deg) ` +
				`translateX(${rotationProps.translateX}) ` +
				`translateY(${rotationProps.translateY}) ` +
				`translateZ(0)`;
			element.style.transformOrigin = element.style.webkitTransformOrigin = `top left`;
		}
	}
}
