import { EventEmitter } from 'events';
import IVideoPlayer, { IPrepareOptions } from '../../Video/IVideoPlayer';
import IVideoEventEmitter from '../../Video/IVideoEventEmitter';
import IVideoEvent from '../../Video/IVideoEvent';
import { convertClientCoordinatesToSystem } from './cssRotationHelper';

/**
 * Wrapper class for any implementation of video player that converts CSS-rotated coordinates back to landscape for the system
 *
 * When the content is rotated via CSS, all the coordinates that are sent from it are rotated as well.
 * That's bad because from the point of view of the system it's landscape and the rotated coordinates don't make sense.
 * This is a proxy class that converts the coordinates between the client and the system so neither of them have to worry about it.
 */
export default class CSSRotationVideoPlayerAdapter implements IVideoPlayer {
	constructor(
		private window: Window,
		private videoPlayer: IVideoPlayer,
		private getAngle: () => Promise<number>,
	) {}

	public getMaxVideoCount(): number | null {
		return this.videoPlayer.getMaxVideoCount();
	}

	public async prepare(uri: string, x: number, y: number, width: number, height: number, options?: IPrepareOptions): Promise<void> {
		const angle = await this.getAngle();
		const coords = convertClientCoordinatesToSystem(this.window, angle, x, y, width, height);
		await this.videoPlayer.prepare(uri, coords.x, coords.y, coords.width, coords.height, options);
	}

	public async play(uri: string, x: number, y: number, width: number, height: number): Promise<IVideoEventEmitter> {
		const angle = await this.getAngle();
		const coords = convertClientCoordinatesToSystem(this.window, angle, x, y, width, height);
		const video = await this.videoPlayer.play(uri, coords.x, coords.y, coords.width, coords.height);
		return this.convertEventEmitterWithConvertedCoordinatesBackToOriginalCoordinates(video, x, y, width, height);
	}

	public async stop(uri: string, x: number, y: number, width: number, height: number): Promise<void> {
		const angle = await this.getAngle();
		const coords = convertClientCoordinatesToSystem(this.window, angle, x, y, width, height);
		await this.videoPlayer.stop(uri, coords.x, coords.y, coords.width, coords.height);
	}

	public async pause(uri: string, x: number, y: number, width: number, height: number): Promise<void> {
		const angle = await this.getAngle();
		const coords = convertClientCoordinatesToSystem(this.window, angle, x, y, width, height);
		await this.videoPlayer.pause(uri, coords.x, coords.y, coords.width, coords.height);
	}

	public async resume(uri: string, x: number, y: number, width: number, height: number): Promise<void> {
		const angle = await this.getAngle();
		const coords = convertClientCoordinatesToSystem(this.window, angle, x, y, width, height);
		await this.videoPlayer.resume(uri, coords.x, coords.y, coords.width, coords.height);
	}

	public clearAll(): Promise<void> {
		return this.videoPlayer.clearAll();
	}

	public async getDuration(uri: string, x: number, y: number, width: number, height: number) {
		const angle = await this.getAngle();
		const coords = convertClientCoordinatesToSystem(this.window, angle, x, y, width, height);
		return this.videoPlayer.getDuration(uri, coords.x, coords.y, coords.width, coords.height);
	}

	private convertEventEmitterWithConvertedCoordinatesBackToOriginalCoordinates(
		videoEmitter: IVideoEventEmitter,
		originalX: number,
		originalY: number,
		originalWidth: number,
		originalHeight: number,
	): IVideoEventEmitter {
		const convertedVideoEmitter = new EventEmitter();
		const convertEvent = (event: IVideoEvent) => ({
			...event,
			srcArguments: {
				uri: event.srcArguments.uri,
				x: originalX,
				y: originalY,
				width: originalWidth,
				height: originalHeight,
			},
		});

		videoEmitter.on('playing', (event: IVideoEvent) => convertedVideoEmitter.emit('playing', convertEvent(event)));
		videoEmitter.on('ended', (event: IVideoEvent) => convertedVideoEmitter.emit('ended', convertEvent(event)));
		videoEmitter.on('error', (event: IVideoEvent) => convertedVideoEmitter.emit('error', convertEvent(event)));
		videoEmitter.on('stopped', (event: IVideoEvent) => convertedVideoEmitter.emit('stopped', convertEvent(event)));

		// "error" event type is treated as a special case and has to have at least one listener or it can crash the whole process
		// https://nodejs.org/api/events.html#events_error_events
		convertedVideoEmitter.on('error', () => {
			/* do nothing */
		});

		return convertedVideoEmitter;
	}
}
