import { EventEmitter } from 'events';
import IStreamPlayer, { IStreamOptions, ITrackInfo, TrackType } from '../../Stream/IStreamPlayer';
import IStream from '../../Stream/IStream';
import { convertClientCoordinatesToSystem } from './cssRotationHelper';
import IStreamEvent from '../../Stream/IStreamEvent';
import { IStreamTracksChangedEvent } from '../../Stream/streamEvents';
import { IVideoSrcArguments } from '../../Video/IVideoSrcArguments';

/**
 * Wrapper class for any implementation of stream player that converts CSS-rotated coordinates back to landscape for the system
 *
 * When the content is rotated via CSS, all the coordinates that are sent from it are rotated as well.
 * That's bad because from the point of view of the system it's landscape and the rotated coordinates don't make sense.
 * This is a proxy class that converts the coordinates between the client and the system so neither of them have to worry about it.
 */
export default class CSSRotationStreamPlayerAdapter implements IStreamPlayer {
	constructor(
		private window: Window,
		private streamPlayer: IStreamPlayer,
		private getAngle: () => Promise<number>,
	) {}

	public async prepare(uri: string, x: number, y: number, width: number, height: number, options?: IStreamOptions): Promise<void> {
		const angle = await this.getAngle();
		const coords = convertClientCoordinatesToSystem(this.window, angle, x, y, width, height);
		await this.streamPlayer.prepare(uri, coords.x, coords.y, coords.width, coords.height, options);
	}

	public async play(uri: string, x: number, y: number, width: number, height: number, options?: IStreamOptions): Promise<IStream> {
		const angle = await this.getAngle();
		const coords = convertClientCoordinatesToSystem(this.window, angle, x, y, width, height);
		const stream = await this.streamPlayer.play(uri, coords.x, coords.y, coords.width, coords.height, options);
		return this.convertEventEmitterWithConvertedCoordinatesBackToOriginalCoordinates(stream, x, y, width, height);
	}

	public async stop(uri: string, x: number, y: number, width: number, height: number): Promise<void> {
		const angle = await this.getAngle();
		const coords = convertClientCoordinatesToSystem(this.window, angle, x, y, width, height);
		await this.streamPlayer.stop(uri, coords.x, coords.y, coords.width, coords.height);
	}

	public async pause(uri: string, x: number, y: number, width: number, height: number): Promise<void> {
		const angle = await this.getAngle();
		const coords = convertClientCoordinatesToSystem(this.window, angle, x, y, width, height);
		await this.streamPlayer.pause(uri, coords.x, coords.y, coords.width, coords.height);
	}

	public async resume(uri: string, x: number, y: number, width: number, height: number): Promise<void> {
		const angle = await this.getAngle();
		const coords = convertClientCoordinatesToSystem(this.window, angle, x, y, width, height);
		await this.streamPlayer.resume(uri, coords.x, coords.y, coords.width, coords.height);
	}

	public async clearAll(): Promise<void> {
		return this.streamPlayer.clearAll();
	}

	public async getTracks(videoId: IVideoSrcArguments): Promise<ITrackInfo[]> {
		return this.streamPlayer.getTracks(videoId);
	}

	public async selectTrack(videoId: IVideoSrcArguments, trackType: TrackType, groupId: string, trackIndex: number): Promise<any> {
		return this.streamPlayer.selectTrack(videoId, trackType, groupId, trackIndex);
	}

	public async resetTrack(videoId: IVideoSrcArguments, trackType: TrackType, groupId?: string): Promise<any> {
		return this.streamPlayer.resetTrack(videoId, trackType, groupId);
	}

	private convertEventEmitterWithConvertedCoordinatesBackToOriginalCoordinates(
		streamEmitter: IStream,
		originalX: number,
		originalY: number,
		originalWidth: number,
		originalHeight: number,
	): IStream {
		const convertedStreamEmitter = new EventEmitter();
		const convertEvent = (event: IStreamEvent<any>) => ({
			...event,
			uri: event.uri,
			x: originalX,
			y: originalY,
			width: originalWidth,
			height: originalHeight,
			options: event.options,
		});

		streamEmitter.on('error', (event: IStreamEvent<any>) => convertedStreamEmitter.emit('error', convertEvent(event)));
		streamEmitter.on('connected', (event: IStreamEvent<any>) => convertedStreamEmitter.emit('connected', convertEvent(event)));
		streamEmitter.on('disconnected', (event: IStreamEvent<any>) => convertedStreamEmitter.emit('disconnected', convertEvent(event)));
		streamEmitter.on('closed', (event: IStreamEvent<any>) => convertedStreamEmitter.emit('closed', convertEvent(event)));
		streamEmitter.on('tracks_changed', (event: IStreamTracksChangedEvent) =>
			convertedStreamEmitter.emit('tracks_changed', convertEvent(event)),
		);

		// "error" event type is treated as a special case and has to have at least one listener or it can crash the whole process
		// https://nodejs.org/api/events.html#events_error_events
		convertedStreamEmitter.on('error', () => {
			/* do nothing */
		});

		return convertedStreamEmitter;
	}
}
