import PowerActionType from '@signageos/actions/dist/Device/Power/PowerActionType';
import Weekday from '@signageos/actions/dist/Misc/Weekday';
import Property from '../../Property/Property';
import { IPropertyStorage } from '../../Property/propertyStorage';
import IPowerActionTimerStorage from './IPowerActionTimerStorage';
import { PowerActionRule, PowerActionRules } from './IPowerActionTimer';

type StoredPowerActionRule = {
	powerType: string;
	weekdays: string[];
	time: string;
};

type StoredPowerActionRules = { [uid: string]: StoredPowerActionRule };

export type PowerActionRulesProperty = Property.POWER_ACTIONS_SCHEDULE_SETTINGS_FRONT | Property.POWER_ACTIONS_SCHEDULE_SETTINGS_MANAGEMENT;

type PrivatePowerActionRulesProperty = PowerActionRulesProperty | Property.POWER_ACTIONS_SCHEDULE_SETTINGS;
export default class PowerActionTimerPropertyStorage implements IPowerActionTimerStorage {
	constructor(
		private readonly propertyStorage: IPropertyStorage,
		private readonly property: PrivatePowerActionRulesProperty,
	) {}

	public async content(): Promise<PowerActionRules> {
		const currentSettings = await this.loadSettings();
		return Object.keys(currentSettings).reduce(
			(result: PowerActionRules, uid: string) => ({
				...result,
				[uid]: {
					powerType: PowerActionType[currentSettings[uid].powerType as keyof typeof PowerActionType],
					weekdays: currentSettings[uid].weekdays.map((weekday: keyof typeof Weekday) => Weekday[weekday]),
					time: currentSettings[uid].time,
				},
			}),
			{},
		);
	}

	public async put(uid: string, rule: PowerActionRule): Promise<void> {
		const { powerType, weekdays, time } = rule;
		const currentSettings = await this.loadSettings();
		if (currentSettings[uid]) {
			console.warn('Scheduled power action with uid already exists: ' + uid);
		}
		await this.saveSettings({
			...currentSettings,
			[uid]: {
				powerType: PowerActionType[powerType],
				weekdays: weekdays.map((weekday: Weekday) => Weekday[weekday]),
				time,
			},
		});
	}

	public async remove(uid: string): Promise<boolean> {
		const currentSettings = await this.loadSettings();
		if (!currentSettings[uid]) {
			return false;
		}
		delete currentSettings[uid];
		await this.saveSettings(currentSettings);
		return true;
	}

	public async clear(): Promise<boolean> {
		const currentSettings = await this.loadSettings();
		if (Object.keys(currentSettings).length === 0) {
			return false;
		}
		await this.saveSettings({});
		return true;
	}

	public async putAll(rules: PowerActionRules) {
		const currentSettings = await this.loadSettings();
		const storedRules: StoredPowerActionRules = Object.keys(rules).reduce(
			(result: StoredPowerActionRules, uid: string) => ({
				...result,
				[uid]: {
					powerType: PowerActionType[rules[uid].powerType],
					weekdays: rules[uid].weekdays.map((weekday: Weekday) => Weekday[weekday]),
					time: rules[uid].time,
				},
			}),
			{},
		);
		await this.saveSettings({
			...currentSettings,
			...storedRules,
		});
		return true;
	}

	/**
	 * @returns Whether this storage has ever persisted any data.
	 */
	public async hasPersistedData(): Promise<boolean> {
		const currentSettings = await this.propertyStorage.getValueOrDefault<any>(this.property, undefined);
		return typeof currentSettings !== 'undefined';
	}

	private async loadSettings(): Promise<StoredPowerActionRules> {
		return await this.propertyStorage.getValueOrDefault<StoredPowerActionRules>(this.property, {});
	}

	private async saveSettings(rules: StoredPowerActionRules): Promise<void> {
		return await this.propertyStorage.setValue<StoredPowerActionRules>(this.property, rules);
	}
}
