import ScreenRotationManager from '../Screen/ScreenRotationManager';
import IframeBrowser from '../../Browser/IframeBrowser';
import DefaultSettingsManager from './DefaultSettingsManager';
import IBrowser from '../IBrowser';
import ProprietaryStreamPlayer from '../../Stream/ProprietaryStreamPlayer';
import IVideoPlayer from '../../Video/IVideoPlayer';
import IStreamPlayer from '../../Stream/IStreamPlayer';
import HTMLVideoPlayer from '../../Video/HTMLVideoPlayer';
import ReconnectStreamPlayer from '../../Stream/ReconnectStreamPlayer';
import ScreenRotationViewportPropsProvider from '../Screen/ScreenRotationViewportPropsProvider';
import { DefaultBasicDriver } from './DefaultBasicDriver';
import DefaultCacheDriver from './DefaultCacheDriver';
import { DefaultFrontDriver } from './DefaultFrontDriver';
import { DefaultManagementDriver } from './DefaultManagementDriver';
import CategorizedLocalStorage from '../../Storage/CategorizedLocalStorage';
import { getEmulatorUid } from './getEmulatorUid';
import { createBrowserFileSystem } from '../../FileSystem/browserFileSystemFactory';

const MAX_ALLOWED_VIDEOS = 4;

/**
 * @param storageBaseUrl The Base URL of where the cacheStorage is located (prefixed) accessible for serviceWorker.
 * There is a limitation for the scope of serviceWorker that it can by max in the same directory as the serviceWorker JS file.
 * So all cached files has to be stored inside the same directory or subdirectories. Otherwise the cache wouldn't work.
 * By default, it's set to current origin but it should be rather filled anyway.
 */
export function createDefaultDrivers(window: Window, storageBaseUrl: string = window.location.origin) {
	const foregroundVideosWrapperElement = createForegroundVideosWrapperElement(window);
	const backgroundVideosWrapperElement = createBackgroundVideosWrapperElement(window);

	const settingsManager = new DefaultSettingsManager(window);

	const screenRotationManager = createScreenRotationManager(window, settingsManager, [
		foregroundVideosWrapperElement,
		backgroundVideosWrapperElement,
	]);
	const browser = createBrowser(window, screenRotationManager);
	const { videoPlayer, streamPlayer } = createVideoAndStreamPlayers(
		window,
		screenRotationManager,
		settingsManager,
		foregroundVideosWrapperElement,
		backgroundVideosWrapperElement,
	);

	const deviceUid = getEmulatorUid(window.location.href);

	const realFileSystem = createBrowserFileSystem(window, deviceUid, storageBaseUrl);
	const storage = new CategorizedLocalStorage(window.localStorage, deviceUid);

	const basicDriver = new DefaultBasicDriver(window, deviceUid);
	const cacheDriver = new DefaultCacheDriver(storage);

	const frontDriver = new DefaultFrontDriver(window, videoPlayer, streamPlayer, browser, basicDriver, cacheDriver, storage, realFileSystem);
	const managementDriver = new DefaultManagementDriver(
		window,
		deviceUid,
		settingsManager,
		screenRotationManager,
		cacheDriver,
		basicDriver,
		realFileSystem,
	);

	return { frontDriver, managementDriver, cacheDriver };
}

function createForegroundVideosWrapperElement(window: Window) {
	const HTML_ELEMENT_ID = 'videos-wrapper-foreground';
	const videosWrapper = window.document.createElement('div');
	videosWrapper.id = HTML_ELEMENT_ID;
	videosWrapper.setAttribute('class', 'video-wrapper');
	window.document.body.appendChild(videosWrapper);
	return videosWrapper;
}

function createBackgroundVideosWrapperElement(window: Window) {
	const HTML_ELEMENT_ID = 'videos-wrapper-background';
	const videosWrapper = window.document.createElement('div');
	videosWrapper.id = HTML_ELEMENT_ID;
	videosWrapper.setAttribute('class', 'video-wrapper');
	window.document.body.appendChild(videosWrapper);
	return videosWrapper;
}

function createScreenRotationManager(window: Window, settingsManager: DefaultSettingsManager, videosWrapperElements: HTMLElement[]) {
	const bodyElement = window.document.getElementById('body')!;
	const elementsToRotate: HTMLElement[] = [bodyElement];
	if (!isFakePortrait(settingsManager)) {
		elementsToRotate.push(...videosWrapperElements);
	}
	const propsProvider = new ScreenRotationViewportPropsProvider(false);
	return new ScreenRotationManager(window, propsProvider, elementsToRotate, async () => settingsManager.getOrientation());
}

function createBrowser(window: Window, screenRotationManager: ScreenRotationManager): IBrowser {
	const browser = new IframeBrowser(window);
	return screenRotationManager.adaptHTMLElementProducer(browser) as IframeBrowser;
}

function createVideoAndStreamPlayers(
	window: Window,
	screenRotationManager: ScreenRotationManager,
	settingsManager: DefaultSettingsManager,
	foregroundVideosWrapperElement: HTMLElement,
	backgroundVideosWrapperElement: HTMLElement,
): { videoPlayer: IVideoPlayer; streamPlayer: IStreamPlayer } {
	const videoPlayer = new HTMLVideoPlayer(
		MAX_ALLOWED_VIDEOS,
		window,
		foregroundVideosWrapperElement,
		backgroundVideosWrapperElement,
		async () => settingsManager.getVolume(),
	);
	settingsManager.onVolumeChange((volume: number) => videoPlayer.setVolume(volume));

	const streamPlayer = new ReconnectStreamPlayer(new ProprietaryStreamPlayer(videoPlayer));

	if (isFakePortrait(settingsManager)) {
		return {
			videoPlayer: screenRotationManager.adaptVideoPlayer(videoPlayer),
			streamPlayer: screenRotationManager.adaptStreamPlayer(streamPlayer),
		};
	} else {
		return { videoPlayer, streamPlayer };
	}
}

function isFakePortrait(settingsManager: DefaultSettingsManager) {
	const videoOrientation = settingsManager.getVideoOrientation();
	return videoOrientation !== null;
}
