import Debug from 'debug';
import { createFileSystemWithReservedSpaceWithWindowHttpHeadFetcher } from '../../FileSystem/FileSystemWithReservedSpace';
import { RESERVED_SPACE_PERCENTAGE } from '../../FileSystem/fileSystemHelper';
import ISecurity from '../../Front/Security/ISecurity';
import { createCompatibleSecurity } from '../../Front/Security/security';
import createPropertyStorage from '../../Property/propertyStorage';
import { updateOverlay } from '../../Screen/overlayHelper';
import AsyncStorage from '../../Storage/Async/AsyncStorage';
import { asynchronizeStorage } from '../../Storage/Async/asyncStorageHelper';
import IStreamPlayer from '../../Stream/IStreamPlayer';
import { IProprietaryTimerStorage } from '../../Timer/ITimerStorage';
import ProprietaryTimerPropertyStorage from '../../Timer/ProprietaryTimerPropertyStorage';
import IVideoPlayer from '../../Video/IVideoPlayer';
import FrontCapability from '../Front/FrontCapability';
import IFrontDriver, { Hardware } from '../Front/IFrontDriver';
import ITouchEventMessage from '../Front/ITouchEventMessage';
import NotImplementedSerial from '../Hardware/NotImplementedSerial';
import IBasicDriver from '../IBasicDriver';
import IBrowser from '../IBrowser';
import ICacheDriver from '../ICacheDriver';
import IFileSystem from '../IFileSystem';
import ISignature from '../ISignature';
import IKeyUpEvent from '../Input/IKeyUpEvent';
import { keyboardKeyMap } from '../Input/KeyCode';

const debug = Debug('@signageos/front-display:NativeDevice:Default:DefaultFrontDriver');

export class DefaultFrontDriver implements IFrontDriver {
	private static BRIGHTNESS_KEY: string = 'default.native_device.BRIGHTNESS';

	public readonly hardware: Hardware = {
		led: {
			async setColor(_color: string) {
				console.info(new Error('Not implemented hardware led set color'));
			},
		},
		serial: new NotImplementedSerial(),
		barcodeScanner: {
			async getVersion() {
				throw new Error('Not implemented barcode scanner');
			},
			async startScanning() {
				throw new Error('Not implemented barcode scanner');
			},
			async stopScanning() {
				throw new Error('Not implemented barcode scanner');
			},
		},
	};

	public readonly fileSystem: IFileSystem;
	public readonly security: ISecurity;
	public readonly proprietaryTimerStorage: IProprietaryTimerStorage;

	private asyncStorage: AsyncStorage;

	public static async support(_window: Window) {
		return true;
	}

	constructor(
		private window: Window,
		public readonly video: IVideoPlayer,
		public readonly stream: IStreamPlayer,
		public readonly browser: IBrowser,
		private basicDriver: IBasicDriver,
		cacheDriver: ICacheDriver,
		private storage: Storage,
		private realFileSystem: IFileSystem & { initialize(): Promise<void> },
	) {
		this.asyncStorage = asynchronizeStorage(this.storage);
		this.fileSystem = createFileSystemWithReservedSpaceWithWindowHttpHeadFetcher(
			this.realFileSystem,
			this.window,
			RESERVED_SPACE_PERCENTAGE,
		);

		this.security = createCompatibleSecurity(
			() => this,
			() => cacheDriver,
		);
		this.proprietaryTimerStorage = new ProprietaryTimerPropertyStorage(createPropertyStorage(() => cacheDriver));
	}

	public async getConfigurationBaseUrl() {
		return this.basicDriver.getConfigurationBaseUrl();
	}

	public getApplicationType() {
		return this.basicDriver.getApplicationType();
	}

	public async frontSupports(capability: FrontCapability) {
		switch (capability) {
			case FrontCapability.FILE_SYSTEM_INTERNAL_STORAGE:
			case FrontCapability.FILE_SYSTEM_FILE_CHECKSUM:
			case FrontCapability.TIMERS_PROPRIETARY:
			case FrontCapability.VIDEO_4K:
			case FrontCapability.FRONT_OSD:
				return true;
			case FrontCapability.BROWSER:
				return this.browser.isSupported();
			default:
				return false;
		}
	}

	public async initialize(_staticBaseUrl: string) {
		await updateOverlay(this.window, this.asyncStorage);
		this.updateBrightness();
		await this.realFileSystem.initialize();
	}

	public isDetected() {
		return false;
	}

	public start() {
		this.basicDriver.start();
	}

	public stop() {
		this.basicDriver.stop();
	}

	public bindKeyUp(keyUpListener: (keyUpEvent: IKeyUpEvent) => void) {
		this.window.addEventListener('keyup', (event: KeyboardEvent) => {
			const keyCode = keyboardKeyMap[event.key] ?? keyboardKeyMap[event.key.toLocaleLowerCase()];
			if (keyCode !== undefined) {
				keyUpListener({ keyCode });
			} else {
				debug(new Error('Not supported keyCode ' + event.key));
			}
		});
	}

	public async getDeviceUid() {
		return this.basicDriver.getDeviceUid();
	}

	public async isConnected() {
		return this.basicDriver.isConnected();
	}

	public async getSessionId(sessionIdKey: string) {
		return this.basicDriver.getSessionId(sessionIdKey);
	}

	public async setSessionId(sessionIdKey: string, sessionId: string) {
		return this.basicDriver.setSessionId(sessionIdKey, sessionId);
	}

	public remoteControlSetEnabled(_enabled: boolean) {
		console.info('Not implemented set on/off remote control');
		return Promise.resolve();
	}

	public async restoreDisplayArea() {
		await this.video.clearAll();
		await this.stream.clearAll();
		await this.browser.close();
	}

	public remoteControlIsEnabled() {
		console.info('Not implemented get on/off remote control');
		return Promise.resolve(true);
	}

	public async controlSetPin(_pin: string) {
		console.warn(new Error('Not implemented set control pin'));
	}

	public async browserOpenLink(uri: string) {
		this.window.open(uri, '_blank', 'location=yes,height=640,width=480,scrollbars=yes,status=yes');
	}

	public async getCurrentSignature(): Promise<ISignature | null> {
		return null;
	}

	public getOSDUri(): string {
		return 'osd/index.html';
	}

	public async forwardTouchEvents(_callback: (data: ITouchEventMessage) => void) {
		return undefined;
	}

	private updateBrightness() {
		const brightness = this.window.localStorage.getItem(DefaultFrontDriver.BRIGHTNESS_KEY);
		if (brightness !== null) {
			const newFilter = `brightness(${50 + parseInt(brightness) / 2}%)`;
			this.window.document.body.style.filter = newFilter;
			this.window.document.body.style.webkitFilter = newFilter;
		}
	}
}
