import { locked } from '@signageos/lib/es6/Lock/lockedDecorator';
import AsyncStorage from '../../Storage/Async/AsyncStorage';

interface IActionTimerConfig {
	callback: () => Promise<void>;
	timeoutPointer?: NodeJS.Timeout;
}

interface IStoredActionTimerConfig {
	executeAt: number;
}

export default class ActionTimer {
	private static MINIMAL_REACTION_PERIOD_MS: number = 1e3;
	private timer: IActionTimerConfig;

	constructor(
		private storage: AsyncStorage,
		private key: string,
		callback: () => Promise<void>,
	) {
		this.timer = { callback };
	}

	@locked('actionTimer', { scope: 'instance' })
	public async init() {
		const rawConfiguration = await this.storage.getItem(this.getStorageKey());
		if (rawConfiguration) {
			const { executeAt } = this.parseConfiguration(rawConfiguration);
			if (executeAt - Date.now() > ActionTimer.MINIMAL_REACTION_PERIOD_MS) {
				this.start(executeAt - Date.now());
			}
		}
	}

	@locked('actionTimer', { scope: 'instance' })
	public async set(timeout: number) {
		this.stop();
		await this.storage.setItem(this.getStorageKey(), this.stringifyConfiguration(timeout));
		this.start(timeout);
	}

	@locked('actionTimer', { scope: 'instance' })
	public async drop() {
		this.stop();
		await this.storage.removeItem(this.getStorageKey());
	}

	private start(timeout: number) {
		this.timer.timeoutPointer = setTimeout(() => {
			this.drop();
			this.timer?.callback();
		}, timeout);
	}

	private stop() {
		if (this.timer.timeoutPointer) {
			clearTimeout(this.timer.timeoutPointer);
		}
	}

	private getStorageKey() {
		return `default.native_device.ACTION_TIMERS.${this.key}`;
	}

	private stringifyConfiguration(timeout: number) {
		return JSON.stringify({ executeAt: Date.now() + timeout });
	}

	private parseConfiguration(rawConfiguration: string) {
		return JSON.parse(rawConfiguration) as IStoredActionTimerConfig;
	}
}
