import PowerActionType from '@signageos/actions/dist/Device/Power/PowerActionType';
import { cloneDeep } from 'lodash';
import { Store } from 'redux';
import { createApplication } from '../Application/applicationFactory';
import { SubscriptionType } from '../Display/IConfig';
import { createAbsentResponsibilities } from '../Feature/Responsibilities';
import Responsibility from '../Feature/Responsibility';
import { IBundledApplet } from '../Front/Applet/BundledApplet/bundledAppletActions';
import { fetch } from '../Isomorphic/fetch';
import { ManagementCacheDriver, normalizeManagementCacheDriver } from '../NativeDevice/Default/combinedDriver';
import IManagementDriver from '../NativeDevice/Management/IManagementDriver';
import createPropertyStorage from '../Property/propertyStorage';
import IBundledServlet from './Servlet/IBundledServlet';
import { IManagementState, managementReducer } from './managementReducers';
import { managementSaga } from './managementSagas';

function startBackupRestartChecking(managementDriver: IManagementDriver, defaultBaseUrl: string) {
	const BACKUP_RESTART_STATUS_CODE = 299;
	setInterval(async () => {
		const type = PowerActionType[PowerActionType.BACKUP_RESTART];
		const deviceUid = await managementDriver.getDeviceUid();
		const powerActionUri = defaultBaseUrl + '/power-action/' + type + '/' + deviceUid;
		const response = await fetch(powerActionUri);
		if (response.status === BACKUP_RESTART_STATUS_CODE) {
			await fetch(powerActionUri, { method: 'POST' });
			await managementDriver.systemReboot();
		}
	}, 200e3);
}

export interface IManagementOptions {
	baseUrl: string;
	platformUri: string;
	staticBaseUrl: string;
	uploadBaseUrl: string;
	weinreServerUrl: string;
	extendedManagementUrl: string | null;
	pingUrl: string;
	checkInterval: number;
	publicKey: string;
	sessionIdKey: string;
	nativeDriver: ManagementCacheDriver;
	frontDisplayVersion: string;
	applicationVersion: string;
	subscriptionType: SubscriptionType;
	bundledApplet: null | IBundledApplet;
	bundledServlet: IBundledServlet | null;
	autoVerification: { organizationUid: string; deviceName?: string } | undefined;
	excludedResponsibilities: Responsibility[];
}

export default async (options: IManagementOptions) => {
	const { managementDriver, cacheDriver } = normalizeManagementCacheDriver(options.nativeDriver);

	const responsibilities = createAbsentResponsibilities(...options.excludedResponsibilities);
	if (responsibilities.has(Responsibility.POWER)) {
		startBackupRestartChecking(managementDriver, options.baseUrl);
	}

	const propertyStorage = createPropertyStorage(() => cacheDriver);
	const subscriptionType = options.subscriptionType;
	const { excludedResponsibilities: _, ...newOptions } = options;
	const {
		baseUrl,
		platformUri,
		staticBaseUrl,
		uploadBaseUrl,
		checkInterval,
		weinreServerUrl: weinreUri,
		extendedManagementUrl,
		...sagaOptions
	} = {
		responsibilities,
		...newOptions,
	};
	const config = { baseUrl, platformUri, staticBaseUrl, uploadBaseUrl, weinreUri, extendedManagementUrl, subscriptionType, checkInterval };
	let lastSagaError: Error | undefined;
	const { store, storeEmitter } = await createApplication(
		managementReducer,
		(storage: Store<IManagementState>) => managementSaga(config, sagaOptions, storage.getState, propertyStorage),
		() => managementDriver,
		(e: Error) => {
			lastSagaError = cloneDeep(e);
			console.warn('Sagas broken', e);
		},
	);

	if (typeof window !== 'undefined') {
		(window as any).__management = { ...options, store, storeEmitter, lastSagaError };
	}
};
