import * as t from './../../Test/TestFramework';
import IManagementDriver from '../../NativeDevice/Management/IManagementDriver';
import should from '../../Test/should';
import ManagementCapability from '../../NativeDevice/Management/ManagementCapability';
import { TestCase } from '@signageos/actions/dist/Device/Test/deviceTestActions';
import * as momentTimezone from 'moment-timezone';

export default async (nativeDriver: IManagementDriver) => {
	const skipIfDeviceDoesntSupport = async (...capabilities: ManagementCapability[]) => {
		for (const capability of capabilities) {
			if (!(await nativeDriver.managementSupports(capability))) {
				t.skip(`The device does not provide "${capability}" capability`);
				break;
			}
		}
	};

	return t.describe(TestCase.DATE_TIME, function* () {
		yield t.it('Should get datetime and timezone', async () => {
			await skipIfDeviceDoesntSupport(ManagementCapability.GET_TIMEZONE);

			const timestampMs = await nativeDriver.timeManager.getEpochMillis();
			const timezone = await nativeDriver.timeManager.getZoneId();
			should(timestampMs).be.Number();
			should(timestampMs).be.greaterThanOrEqual(0);
			should(timezone).be.oneOf(momentTimezone.tz.names());
		});

		yield t.it('Should set time', async () => {
			await skipIfDeviceDoesntSupport(ManagementCapability.SET_TIME, ManagementCapability.GET_TIMEZONE);

			const timestampMs = await nativeDriver.timeManager.getEpochMillis();
			const timezone = await nativeDriver.timeManager.getZoneId();

			const diff = 5 * 60 * 1000; // 5 minutes
			await nativeDriver.timeManager.setManual(timestampMs - diff, timezone);

			const newTimestampMs = await nativeDriver.timeManager.getEpochMillis();
			const newTimezone = await nativeDriver.timeManager.getZoneId();
			should(newTimezone).equal(timezone);
			should(newTimestampMs - timestampMs).be.greaterThanOrEqual(diff);

			await nativeDriver.timeManager.setManual(timestampMs, timezone);
		});

		yield t.it('Should change the timezone', async () => {
			await skipIfDeviceDoesntSupport(ManagementCapability.SET_TIME, ManagementCapability.GET_TIMEZONE, ManagementCapability.SET_TIMEZONE);

			const timestampMs = await nativeDriver.timeManager.getEpochMillis();
			const timezone = await nativeDriver.timeManager.getZoneId();
			const timezoneList = momentTimezone.tz.names();
			const diff = 5 * 60 * 1000; // 5 minutes
			await nativeDriver.timeManager.setManual(
				timestampMs - diff,
				timezoneList[(timezoneList.indexOf(timezone) + 10) % timezoneList.length],
			);

			const newTimestampMs = await nativeDriver.timeManager.getEpochMillis();
			const newTimezone = await nativeDriver.timeManager.getZoneId();
			should(newTimezone).equal(timezoneList[(timezoneList.indexOf(timezone) + 10) % timezoneList.length]);
			should(newTimestampMs - timestampMs).be.greaterThanOrEqual(diff);

			await nativeDriver.timeManager.setManual(timestampMs, timezone);
		});
	});
};
