import { fork, put, takeEvery } from 'redux-saga/effects';
import { VerificationSucceeded } from '@signageos/actions/dist/Device/Verification/deviceVerificationActions';
import { IServlet, SubscribeActiveServlet, UpdateActiveServlets } from '@signageos/actions/dist/Servlet/servletActions';
import { AuthenticationSucceed } from '@signageos/actions/dist/Authentication/authenticationActions';
import { IManagementState } from '../managementReducers';
import { bindWhenPlatform } from '../../Socket/socketActionCreator';
import IManagementDriver from '../../NativeDevice/Management/IManagementDriver';
import ManagementCapability from '../../NativeDevice/Management/ManagementCapability';
import OfflineCache from '../../OfflineCache/OfflineCache';
import HashAlgorithm from '../../NativeDevice/HashAlgorithm';
import { IServletOptions } from '../../Servlet/IServletRunner';
import IBundledServlet from './IBundledServlet';
import { IResponsibilities } from '../../Feature/Responsibilities';
import Responsibility from '../../Feature/Responsibility';
import { whenCapable } from '../../Feature/capable';
import { withDependencies } from '../../DI/dependencyInjection';

export function* subscribeActiveServlets(
	getState: () => IManagementState,
	responsibilities: IResponsibilities,
	getNativeDriver: () => IManagementDriver,
) {
	if (!responsibilities.has(Responsibility.SERVLET)) {
		return;
	}

	yield bindWhenPlatform([AuthenticationSucceed, VerificationSucceeded]);

	const driver = getNativeDriver();
	const supportsServlet: string = yield driver.managementSupports(ManagementCapability.SERVLET);
	if (supportsServlet) {
		yield takeEvery([AuthenticationSucceed, VerificationSucceeded], function* (action: AuthenticationSucceed | VerificationSucceeded) {
			try {
				switch (action.type) {
					case AuthenticationSucceed:
						const state = getState();
						if (state.deviceVerification.verified) {
							yield put({
								type: SubscribeActiveServlet,
							} as SubscribeActiveServlet);
						}
						break;
					case VerificationSucceeded:
						yield put({
							type: SubscribeActiveServlet,
						} as SubscribeActiveServlet);
						break;
					default:
				}
			} catch (error) {
				console.error('subscribeActiveServlets failed', error);
			}
		});
	}
}

export function* runBundledServlet(responsibilities: IResponsibilities, bundledServlet: IBundledServlet | null) {
	if (!responsibilities.has(Responsibility.SERVLET)) {
		return;
	}

	yield fork(
		whenCapable(
			ManagementCapability.SERVLET,
			withDependencies(['managementDriver'], function* ({ managementDriver }) {
				if (bundledServlet) {
					try {
						const options: IServletOptions = {
							env: bundledServlet.env,
							healthcheck: bundledServlet.healthcheck,
						};
						managementDriver.servletRunner.run(bundledServlet.filePath, options);
					} catch (error) {
						console.error('runServlets', error);
					}
				}
			}),
		),
	);
}

export function* runServlets(responsibilities: IResponsibilities, offlineCache: OfflineCache) {
	if (!responsibilities.has(Responsibility.SERVLET)) {
		return;
	}

	yield bindWhenPlatform(UpdateActiveServlets);

	yield fork(
		whenCapable(
			ManagementCapability.SERVLET,
			withDependencies(['staticBaseUrl', 'managementDriver'], function* ({ staticBaseUrl, managementDriver }) {
				yield takeEvery(UpdateActiveServlets, function* (action: UpdateActiveServlets) {
					try {
						yield Promise.all(
							action.servlets.map(async (servlet: IServlet) => {
								await downloadAndExtractServlet(offlineCache, staticBaseUrl, servlet.uid, servlet.version, servlet.checksum);
								const options: IServletOptions = {
									env: servlet.env,
									// TODO add healthcheck option for remote servlets
									//healthcheck: servlet.healthcheck,
								};
								await runServlet(managementDriver, offlineCache, servlet.uid, servlet.version, options);
							}),
						);
					} catch (error) {
						console.error('runServlets', error);
					}
				});
			}),
		),
	);
}

async function downloadAndExtractServlet(
	offlineCache: OfflineCache,
	staticBaseUrl: string,
	servletUid: string,
	servletVersion: string,
	servletChecksum: string,
) {
	const EXTRACT_METHOD = 'zip';
	try {
		const servletDirectoryName = getServletDirectoryName(servletUid, servletVersion);
		const servletArchiveFileName = getServletArchiveFileName(servletUid, servletVersion);
		const servletDirectoryExists = await offlineCache.fileExists(servletDirectoryName);
		if (!servletDirectoryExists) {
			await downloadServletArchiveToLocalStorage(offlineCache, staticBaseUrl, servletUid, servletVersion);
			await validateServletArchiveChecksum(offlineCache, servletArchiveFileName, servletChecksum);
			await offlineCache.extractFile(servletArchiveFileName, servletDirectoryName, EXTRACT_METHOD, true);
			await offlineCache.deleteFile(servletArchiveFileName, false);
		}
	} catch (error) {
		console.error('Error while downloading servlet archive', 'uid: ' + servletUid + ', version: ' + servletVersion, error);
	}
}

async function validateServletArchiveChecksum(offlineCache: OfflineCache, archiveFileName: string, expectedChecksum: string) {
	const CHECKSUM_HASH_ALGORITHM = HashAlgorithm.MD5;
	const servletArchiveChecksum = await offlineCache.getFileChecksum(archiveFileName, CHECKSUM_HASH_ALGORITHM);
	if (servletArchiveChecksum !== expectedChecksum) {
		throw new Error("Checksum of downloaded servlet archive doesn't match expected checksum");
	}
}

async function downloadServletArchiveToLocalStorage(
	offlineCache: OfflineCache,
	staticBaseUrl: string,
	servletUid: string,
	servletVersion: string,
) {
	const servletArchiveFileName = getServletArchiveFileName(servletUid, servletVersion);
	const servletArchiveUri = getServletArchiveDownloadUri(staticBaseUrl, servletUid, servletVersion);
	const DOWNLOAD_RETRIES = 5;
	await offlineCache.retriableDownloadFile(DOWNLOAD_RETRIES, servletArchiveFileName, servletArchiveUri, undefined, true);
	return await offlineCache.getFile(servletArchiveFileName);
}

async function runServlet(
	managementDriver: IManagementDriver,
	offlineCache: OfflineCache,
	servletUid: string,
	servletVersion: string,
	options: IServletOptions,
) {
	const servletDirectory = getServletDirectoryName(servletUid, servletVersion);
	const servletPackageJson = servletDirectory + '/package.json';
	const servletPackageJsonExists = await offlineCache.fileExists(servletPackageJson);
	if (!servletPackageJsonExists) {
		throw new Error('Servlet package is not a valid npm module, missing package.json');
	}
	const servletPackageJsonContents = await offlineCache.readFile(servletPackageJson);
	const mainFileName = JSON.parse(servletPackageJsonContents).main || 'index.js';
	const mainFile = servletDirectory + '/' + mainFileName;
	const mainFilePath = await offlineCache.getFullFilePath(mainFile);
	managementDriver.servletRunner.run(mainFilePath, options);
}

function getServletArchiveFileName(servletUid: string, servletVersion: string) {
	return `servlet/${servletUid}/${servletVersion}.zip`;
}

function getServletDirectoryName(servletUid: string, servletVersion: string) {
	return `servlet/${servletUid}/${servletVersion}`;
}

function getServletArchiveDownloadUri(staticBaseUrl: string, servletUid: string, servletVersion: string) {
	return `${staticBaseUrl}/servlet/${servletUid}/${servletVersion}.zip`;
}
