import { RsaKeyPairData, SecretStorage } from './SecretStorage';
import * as jose from 'jose';
import { KeyLike } from 'jose';
import { JweGeneral, RsaPublicKeyData, SecretManager } from './SecretManager';
import { Buffer } from 'buffer';

const KEY_ALGORITHM = 'RSA-OAEP';
const KEY_SIZE = 4096;
const KEY_HASH = 'SHA-256';

const KEY_ALIAS = 'SecretManager';
const KEY_VALIDITY_MONTHS = 6;

const JWE_ENC = 'A256GCM';

export default class SubtleCryptoSecretManager implements SecretManager {
	constructor(
		private subtleCrypto: () => SubtleCrypto,
		private storage: SecretStorage,
	) {}

	public async generateKeys() {
		const keyPair = await this.subtleCrypto().generateKey(
			{
				name: KEY_ALGORITHM,
				modulusLength: KEY_SIZE,
				publicExponent: new Uint8Array([1, 0, 1]),
				hash: KEY_HASH,
			},
			true,
			['encrypt', 'decrypt'],
		);
		await this.saveKeyPair(keyPair);
	}

	public async encryptUtf8ToJweCompact(text: string): Promise<string> {
		const bytes = Buffer.from(text, 'utf8');
		return this.encryptBytesToJweCompact(bytes);
	}

	public async decryptJweCompactToUtf8(jweCompact: string): Promise<string> {
		const plaintext = await this.decryptJweCompactToBytes(jweCompact);
		return Buffer.from(plaintext).toString('utf-8');
	}

	public async decryptJweCompactToBytes(jweCompact: string): Promise<Uint8Array> {
		const privateKey = await this.requirePrivateKey();
		const { plaintext } = await jose.compactDecrypt(jweCompact, privateKey);
		return plaintext;
	}

	public async encryptBytesToJweCompact(value: Uint8Array): Promise<string> {
		const [publicKey, alg] = await this.requirePublicKey();
		return await new jose.CompactEncrypt(value).setProtectedHeader({ alg, enc: JWE_ENC }).encrypt(publicKey);
	}

	public async encryptUtf8ToJweGeneral(text: string): Promise<JweGeneral> {
		const bytes = Buffer.from(text, 'utf8');
		return this.encryptBytesToJweGeneral(bytes);
	}

	public async decryptJweGeneralToUtf8(jweGeneral: JweGeneral): Promise<string> {
		const plaintext = await this.decryptJweGeneralToBytes(jweGeneral);
		return Buffer.from(plaintext).toString('utf-8');
	}

	public async decryptJweGeneralToBytes(jweGeneral: JweGeneral): Promise<Uint8Array> {
		const privateKey = await this.requirePrivateKey();
		const { plaintext } = await jose.generalDecrypt(jweGeneral, privateKey);
		return plaintext;
	}

	public async encryptBytesToJweGeneral(value: Uint8Array): Promise<JweGeneral> {
		const [publicKey, alg] = await this.requirePublicKey();
		return await new jose.GeneralEncrypt(value)
			.setProtectedHeader({ enc: JWE_ENC })
			.addRecipient(publicKey)
			.setUnprotectedHeader({ alg })
			.encrypt();
	}

	public async getPublicKey(): Promise<RsaPublicKeyData> {
		const keyPair = await this.requireKeyPair();
		return {
			algorithm: keyPair.algorithm,
			publicKeySpki: keyPair.publicKeySpki,
			keyValidity: keyPair.keyValidity,
		};
	}

	private async saveKeyPair(keyPair: CryptoKeyPair) {
		const algorithm = keyPair.publicKey.algorithm as RsaHashedKeyAlgorithm;
		if (!algorithm.name.startsWith('RSA-')) {
			throw new Error(`Unsupported algorithm: ${algorithm.name}`);
		}

		const publicKeySpki = await jose.exportSPKI(keyPair.publicKey);
		const privateKeyPkcs8 = await jose.exportPKCS8(keyPair.privateKey);

		const actualDate = new Date();
		const validUntil = new Date(actualDate);
		validUntil.setMonth(actualDate.getMonth() + KEY_VALIDITY_MONTHS);

		const keyData: RsaKeyPairData = {
			algorithm,
			publicKeySpki,
			privateKeyPkcs8,
			keyValidity: {
				start: actualDate,
				end: validUntil,
			},
		};
		await this.storage.saveKeyPair(KEY_ALIAS, keyData);
	}

	private async requirePrivateKey(): Promise<KeyLike> {
		const keyPair = await this.requireKeyPair();
		const alg = buildJweAlg(keyPair.algorithm);
		return await jose.importPKCS8(keyPair.privateKeyPkcs8, alg);
	}

	private async requirePublicKey(): Promise<[KeyLike, string]> {
		const keyPair = await this.requireKeyPair();
		const alg = buildJweAlg(keyPair.algorithm);
		return [await jose.importSPKI(keyPair.publicKeySpki, alg), alg];
	}

	private async requireKeyPair(): Promise<RsaKeyPairData> {
		const keyPair = await this.storage.getKeyPair(KEY_ALIAS);
		if (!keyPair) {
			throw new Error(`Key not found.`);
		}
		return keyPair;
	}
}

const buildJweAlg = (algorithm: RsaHashedKeyAlgorithm) => {
	switch (algorithm.name) {
		case 'RSA-OAEP':
			const hash = algorithm.hash.name;
			switch (hash) {
				case 'SHA-1':
					return algorithm.name;
				default:
					return `${algorithm.name}-${hash.slice(-3)}`;
			}
		default:
			throw new Error(`Unsupported algorithm: ${algorithm.name}`);
	}
};
