import { DeviceTelemetryType } from '@signageos/common-types/dist/Device/Telemetry/DeviceTelemetryType';
import { runPeriodicTaskSagaUntilSuccessWhilePlatform, runPeriodicTaskSagaWhilePlatform } from '../../../Saga/periodicTaskSaga';
import Debug from 'debug';
import { withDependencies } from '../../../DI/dependencyInjection';
import { fork } from 'redux-saga/effects';

const debug = Debug('@signageos/front-display:Management:Telemetry:deviceTelemetryHelper');

/**
 * Base for initial delay in ms before sending the first telemetry data.
 * On top of this delay there is always a random delay of up to TELEMETRY_INITIAL_RANDOM_DELAY ms.
 */
const TELEMETRY_INITIAL_DELAY = 20e3;
/**
 * Maximum additional random delay in ms before sending the first telemetry data.
 * Related to TELEMETRY_INITIAL_DELAY.
 */
const TELEMETRY_INITIAL_RANDOM_DELAY = 10e3;
/**
 * The period of time in ms of how often will be tried to send the initial telemetry until it's sent at least once.
 */
const TELEMETRY_INITIAL_ONLY_RETRY = 10e3;

/**
 * Calculates the initial delay in ms before sending the first telemetry data.
 * It's a random value between TELEMETRY_INITIAL_DELAY and TELEMETRY_INITIAL_DELAY + TELEMETRY_INITIAL_RANDOM_DELAY.
 */
export function calculateInitialTelemetryTimeoutMs() {
	const initialDelayMs = TELEMETRY_INITIAL_DELAY + Math.round(Math.random() * TELEMETRY_INITIAL_RANDOM_DELAY);
	debug(`calculated initial telemetry timeout ${initialDelayMs}ms`);
	return initialDelayMs;
}

/**
 * The generic function that registers any kind of telemetry.
 *
 * @param type It's currently used only for debug logs. For some old telemetry it can be used plain string because it's not a part of TelemetryType enum yet.
 * @param periodMs How often the telemetry will be sent.
 * Negative values mean the telemetry is disabled.
 * Zero value means the only initial telemetry is sent and then it's disabled.
 * Values greater than zero mean periodic telemetry in milliseconds.
 * @param performTelemetry The function that performs the telemetry.
 */
export const registerTelemetry = <Type extends DeviceTelemetryType, TPerformTelemetry extends () => Generator>(
	type: Type | string,
	periodMs: number,
	performTelemetry: TPerformTelemetry,
) =>
	fork(
		withDependencies(['telemetryInitialTimeoutMs'], function* ({ telemetryInitialTimeoutMs }) {
			if (periodMs > 0) {
				// Values greater than zero mean periodic telemetry in milliseconds.
				yield runPeriodicTaskSagaWhilePlatform(
					telemetryInitialTimeoutMs,
					periodMs,
					function* () {
						try {
							yield* performTelemetry();
						} catch (error) {
							debug(`reporting device telemetry ${type} failed`, error);
						}
					},
					{ randomize: true },
				);
			} else if (periodMs === 0) {
				// Zero value means the only initial telemetry is sent and then it's disabled.
				yield runPeriodicTaskSagaUntilSuccessWhilePlatform(telemetryInitialTimeoutMs, TELEMETRY_INITIAL_ONLY_RETRY, function* () {
					try {
						yield* performTelemetry();
					} catch (error) {
						debug(`reporting device telemetry ${type} failed`, error);
					}
				});
			} else {
				// Negative values mean the telemetry is disabled.
				debug(`reporting device telemetry ${type} disabled by received configuration`);
			}
		}),
	);
