import {
	RegenerateCryptoKeyPair,
	RegenerateCryptoKeyPairFailed,
	RegenerateCryptoKeyPairSucceeded,
} from '@signageos/actions/dist/Device/Secrets/secretActions';
import IManagementDriver from '../../../NativeDevice/Management/IManagementDriver';
import { fork, put } from 'redux-saga/effects';
import { RsaKeyPairData } from '../../Secrets/SecretStorage';
import { withDependencies } from '../../../DI/dependencyInjection';
import { whenCapable } from '../../../Feature/capable';
import ManagementCapability from '../../../NativeDevice/Management/ManagementCapability';
import { RsaPublicKeyData } from '../../Secrets/SecretManager';
import { IResponsibilities } from '../../../Feature/Responsibilities';
import Responsibility from '../../../Feature/Responsibility';
import { bindAndTakeEveryOnlyWhenPlatform } from '../../../Socket/socketActionCreator';
import Property from '../../../Property/Property';

export const CryptographicKeyChanged = 'Secrets.CryptographicKeyChanged';
export interface CryptographicKeyChanged {
	type: typeof CryptographicKeyChanged;
}

export function* regenerateKeyPair(getNativeDriver: () => IManagementDriver, action: RegenerateCryptoKeyPair) {
	const nativeDriver = getNativeDriver();
	try {
		yield nativeDriver.secretManager.generateKeys();
		const keyPairData: RsaKeyPairData = yield nativeDriver.secretManager.getPublicKey();
		if (!keyPairData) {
			throw new Error('Failed to get public key');
		}
		yield put<RegenerateCryptoKeyPairSucceeded>({
			type: RegenerateCryptoKeyPairSucceeded,
			uid: action.uid,
			publicKey: keyPairData.publicKeySpki,
		});
	} catch (error) {
		yield put<RegenerateCryptoKeyPairFailed>({
			type: RegenerateCryptoKeyPairFailed,
			uid: action.uid,
		});
	}
}

export function* cryptographySaga(getNativeDriver: () => IManagementDriver, responsibilities: IResponsibilities) {
	if (!responsibilities.has(Responsibility.SECRET_MANAGEMENT)) {
		return;
	}

	yield bindAndTakeEveryOnlyWhenPlatform(RegenerateCryptoKeyPair, regenerateKeyPair, getNativeDriver);

	yield fork(
		whenCapable(
			ManagementCapability.SECRETS,
			withDependencies(['managementDriver', 'propertyStorage'], function* ({ managementDriver, propertyStorage }) {
				const isGenerated = yield propertyStorage.getValueOrDefault<string | null>(
					Property.LATEST_REPORTED_SETTINGS_CRYPTOGRAPHIC_KEY,
					null,
				);
				if (isGenerated === null) {
					try {
						yield managementDriver.secretManager.generateKeys();
						yield put<CryptographicKeyChanged>({
							type: CryptographicKeyChanged,
						});
					} catch (error) {
						console.error('generateCryptographicKeys failed', error);
					}
				} else {
					yield checkKeyPairValidity();
				}
			}),
		),
	);
}

export function* checkKeyPairValidity() {
	yield fork(
		whenCapable(
			ManagementCapability.SECRETS,
			withDependencies(['managementDriver'], function* ({ managementDriver }) {
				try {
					const publicKeyData: RsaPublicKeyData | any = yield managementDriver.secretManager.getPublicKey();
					if (publicKeyData && publicKeyData.keyValidity && publicKeyData.keyValidity.end) {
						const currentDate = new Date();
						const validityEndDate = new Date(publicKeyData.keyValidity.end);

						if (currentDate >= validityEndDate) {
							try {
								yield managementDriver.secretManager.generateKeys();
								yield put<CryptographicKeyChanged>({
									type: CryptographicKeyChanged,
								});
							} catch (error) {
								console.error('generateCryptographicKeys failed', error);
							}
						}
					}
				} catch (error) {
					console.error('checkKeyPairValidity failed', error);
				}
			}),
		),
	);
}
