import { bindWhenPlatform } from '../../../Socket/socketActionCreator';
import { call, apply, put, takeEvery, fork } from 'redux-saga/effects';
import IManagementDriver from '../../../NativeDevice/Management/IManagementDriver';
import ResolutionList from '../../../NativeDevice/ResolutionList';
import {
	DeviceResize,
	DeviceResizeSucceeded,
	DeviceResizeFailed,
	NotifySupportedResolutions,
} from '@signageos/actions/dist/Device/Resolution/deviceResolutionActions';
import { SocketConnected } from '@signageos/actions/dist/Socket/socketActions';
import ManagementCapability from '../../../NativeDevice/Management/ManagementCapability';
import Debug from 'debug';
import Orientation from '@signageos/common-types/dist/Device/Resolution/Orientation';
import VideoOrientation from '@signageos/common-types/dist/Device/Resolution/VideoOrientation';
import { IResponsibilities } from '../../../Feature/Responsibilities';
import Responsibility from '../../../Feature/Responsibility';
import { withDependencies } from '../../../DI/dependencyInjection';
import { whenCapable } from '../../../Feature/capable';
import processCallback, { ICallback } from '../../../Util/processCallback';

const debug = Debug('@signageos/front-display:Device:Screeb:deviceScreenResizeSagas');

export function* deviceSupportedResolutionsCallback(managementDriver: IManagementDriver) {
	debug('deviceSupportedResolutionsCallback called');
	try {
		const supportedResolutions: ResolutionList = yield managementDriver.getSupportedResolutions();
		debug('supportedResolutions', supportedResolutions);
		yield put({
			type: NotifySupportedResolutions,
			supportedResolutions,
		});
	} catch (error) {
		console.error('automatic getting of supported resolutions not supported', error);
	}
}

export function* deviceResizeCallback(
	managementDriver: IManagementDriver,
	staticBaseUrl: string,
	applicationVersion: string,
	action: DeviceResize,
) {
	try {
		const finalizeCallback: ICallback = yield apply(managementDriver, managementDriver.screenResize, [
			staticBaseUrl,
			Orientation[action.screenOrientation],
			action.screenSize,
			applicationVersion,
			action.videoOrientation ? VideoOrientation[action.videoOrientation] : undefined,
		]);

		yield put({
			type: DeviceResizeSucceeded,
			uid: action.uid,
		} as DeviceResizeSucceeded);
		yield call(processCallback, finalizeCallback);
	} catch (error) {
		console.error('resizeScreen failed', error);
		yield put({
			type: DeviceResizeFailed,
			uid: action.uid,
		} as DeviceResizeFailed);
	}
}

export function* resizeScreen(responsibilities: IResponsibilities) {
	if (!responsibilities.has(Responsibility.SCREEN)) {
		return;
	}

	yield bindWhenPlatform(DeviceResize);

	yield fork(
		whenCapable(
			ManagementCapability.SCREEN_RESIZE,
			withDependencies(
				['applicationVersion', 'managementDriver', 'staticBaseUrl'],
				function* ({ applicationVersion, managementDriver, staticBaseUrl }) {
					yield takeEvery(DeviceResize, deviceResizeCallback, managementDriver, staticBaseUrl, applicationVersion);
				},
			),
		),
	);
}

export function* reportScreen(responsibilities: IResponsibilities) {
	if (!responsibilities.has(Responsibility.SCREEN)) {
		return;
	}

	yield fork(
		whenCapable(
			ManagementCapability.SCREEN_RESIZE,
			withDependencies(['managementDriver'], function* ({ managementDriver }) {
				yield takeEvery(SocketConnected, deviceSupportedResolutionsCallback, managementDriver);
			}),
		),
	);
}
