import { takeEveryAndBindWhenPlatform } from '../../../Socket/socketActionCreator';
import { put, takeEvery } from 'redux-saga/effects';
import IBasicDriver from '../../../NativeDevice/IBasicDriver';
import IManagementDriver from '../../../NativeDevice/Management/IManagementDriver';
import PowerActionType from '@signageos/actions/dist/Device/Power/PowerActionType';
import {
	PerformPowerAction,
	PerformPowerActionSucceed,
	PerformPowerActionFailed,
} from '@signageos/actions/dist/Device/Power/devicePowerActions';
import {
	PerformScheduledPowerAction,
	TriggerDisplayPowerOn,
	TriggerDisplayPowerOff,
} from '../../../Front/Device/Power/devicePowerActionActions';
import ManagementCapability from '../../../NativeDevice/Management/ManagementCapability';
import { IResponsibilities } from '../../../Feature/Responsibilities';
import Responsibility from '../../../Feature/Responsibility';
import { PowerFacade } from './powerFacade';
import { deliver } from '../../../Socket/socketActionDeliverHelper';

export function* devicePerformPowerAction(
	getNativeDriver: () => IManagementDriver & IBasicDriver,
	responsibilities: IResponsibilities,
	powerPerformer: PowerFacade,
) {
	if (!responsibilities.has(Responsibility.POWER)) {
		return;
	}

	yield takeEveryAndBindWhenPlatform(PerformPowerAction, function* ({ powerType, id }: PerformPowerAction) {
		const nativeDriver = getNativeDriver();
		const deviceUid: string = yield nativeDriver.getDeviceUid();
		try {
			const isSupported = yield isPowerTypeSupported(nativeDriver, powerType);
			if (!isSupported) {
				return;
			}
			yield* deliver<PerformPowerActionSucceed>({
				type: PerformPowerActionSucceed,
				id,
				deviceUid,
			} as PerformPowerActionSucceed);
			const powerActionProcessed = yield powerPerformer.performPowerAction(powerType);
			if (!powerActionProcessed) {
				return;
			}
		} catch (error) {
			console.error(error);
			yield put({
				type: PerformPowerActionFailed,
				id,
				deviceUid,
			} as PerformPowerActionFailed);
		}
	});
}

async function isPowerTypeSupported(nativeDriver: IManagementDriver, powerType: PowerActionType) {
	switch (powerType) {
		case PowerActionType.SYSTEM_REBOOT:
			return await nativeDriver.managementSupports(ManagementCapability.SYSTEM_REBOOT);
		case PowerActionType.APP_RESTART:
			return await nativeDriver.managementSupports(ManagementCapability.APP_RESTART);
		case PowerActionType.DISPLAY_POWER_ON:
		case PowerActionType.DISPLAY_POWER_OFF:
			return await nativeDriver.managementSupports(ManagementCapability.DISPLAY_POWER);
		case PowerActionType.FILE_SYSTEM_WIPEOUT:
			return await nativeDriver.managementSupports(ManagementCapability.FILE_SYSTEM_WIPEOUT);
		default:
			return false;
	}
}

export function* devicePerformScheduledPowerAction(powerPerformer: PowerFacade) {
	yield takeEvery(PerformScheduledPowerAction, function* ({ powerType }: PerformScheduledPowerAction) {
		try {
			yield powerPerformer.performPowerAction(powerType);
		} catch (error) {
			console.error('devicePerformScheduledPowerAction failed', error);
		}
	});
}

export function* deviceDisplayPowerAction(getNativeDriver: () => IManagementDriver) {
	yield takeEvery(TriggerDisplayPowerOn, function* () {
		try {
			yield getNativeDriver().displayPowerOn();
		} catch (error) {
			console.error('deviceDisplayPowerAction on failed', error);
		}
	});
	yield takeEvery(TriggerDisplayPowerOff, function* () {
		try {
			yield getNativeDriver().displayPowerOff();
		} catch (error) {
			console.error('deviceDisplayPowerAction off failed', error);
		}
	});
}
