import { put } from 'redux-saga/effects';
import { VerificationSucceeded } from '@signageos/actions/dist/Device/Verification/deviceVerificationActions';
import { AuthenticationSucceed } from '@signageos/actions/dist/Authentication/authenticationActions';
import { IManagementState } from '../../managementReducers';
import {
	ApplyPolicyFailed,
	GetDevicePolicyItems,
	UpdateDevicePolicyItems,
	ApplyPolicySucceeded,
} from '@signageos/actions/dist/Device/Policy/devicePolicyActions';
import { IPropertyStorage } from '../../../Property/propertyStorage';
import Property from '../../../Property/Property';
import IPolicyItem, { isPolicyItemOfType } from '@signageos/common-types/dist/Policy/IPolicyItem';
import { DeviceSettingsType } from '@signageos/common-types/dist/Device/Settings/DeviceSettingsType';
import IPolicyCheckAndSetProperties from './IPolicyCheckAndSetProperties';
import asyncCall from '../../../Util/asyncCall';
import { IResponsibilities } from '../../../Feature/Responsibilities';
import Responsibility from '../../../Feature/Responsibility';
import { bindAndTakeEvery, takeEveryAndBindWhenPlatform } from '../../../Socket/socketActionCreator';
import Debug from 'debug';
import ManagementCapability from '../../../NativeDevice/Management/ManagementCapability';
import { whenResponsible } from '../../../Feature/responsible';
import { whenCapable } from '../../../Feature/capable';
import { withDependencies } from '../../../DI/dependencyInjection';
import IAction from '@signageos/actions/dist/IAction';
import FailsCount from './FailsCount';
import _ from 'lodash';
import { registerTelemetry } from '../Telemetry/deviceTelemetryHelper';
import { PolicyChanged } from '@signageos/common-types/dist/Device/SystemLogs/systemLogs';
import { createSystemLogAction } from '../../../SystemLogs/systemLogActionFactory';

const debug = Debug('@signageos/front-display:Management:Policy');

const POLICY_ACCEPTABLE_FAILS_COUNT = 5;

export enum CheckAndSetResult {
	NOT_CHANGED = 0,
	CHANGED = 1,
}

export function* subscribeDevicePolicy(responsibilities: IResponsibilities, getState: () => IManagementState) {
	if (!responsibilities.has(Responsibility.POLICY)) {
		return;
	}

	yield bindAndTakeEvery([AuthenticationSucceed, VerificationSucceeded], function* (action: AuthenticationSucceed | VerificationSucceeded) {
		try {
			switch (action.type) {
				case AuthenticationSucceed:
					const state = getState();
					if (state.deviceVerification.verified) {
						yield put({
							type: GetDevicePolicyItems,
						} as GetDevicePolicyItems);
					}
					break;
				case VerificationSucceeded:
					yield put({
						type: GetDevicePolicyItems,
					} as GetDevicePolicyItems);
					break;
				default:
			}
		} catch (error) {
			debug('SubscribeDevicePolicy failed', error);
		}
	});
}

export function* updateDevicePolicy(responsibilities: IResponsibilities, propertyStorage: IPropertyStorage) {
	if (!responsibilities.has(Responsibility.POLICY)) {
		return;
	}

	yield takeEveryAndBindWhenPlatform(UpdateDevicePolicyItems, function* (action: UpdateDevicePolicyItems<IPolicyItem>) {
		try {
			const currentPolicyItems: IPolicyItem[] = yield propertyStorage.getValueOrDefault(Property.DEVICE_POLICY, []);
			const changedPolicyItems = _.differenceWith(action.items, currentPolicyItems, _.isEqual);
			yield put(
				createSystemLogAction({
					type: PolicyChanged,
					actual: currentPolicyItems,
					difference: changedPolicyItems,
					updatedAt: new Date().getTime(),
				} satisfies PolicyChanged),
			);
			yield propertyStorage.setValue<IPolicyItem[]>(Property.DEVICE_POLICY, action.items);
			if (action.items.some((item: IPolicyItem) => isPolicyItemOfType(item, DeviceSettingsType.BRIGHTNESS))) {
				yield propertyStorage.removeValue(Property.BRIGHTNESS_SETTINGS);
			}
			for (const proposedPolicyItem of action.items) {
				const currentPolicyItem = currentPolicyItems.find((item: IPolicyItem) => item.type === proposedPolicyItem.type);
				const isPolicyItemChanged = _.isEqual(currentPolicyItem?.value, proposedPolicyItem.value);
				if (!isPolicyItemChanged) {
					const failsCount = new FailsCount(proposedPolicyItem.type, propertyStorage);
					yield failsCount.reset();
				}
			}
		} catch (error) {
			debug('updatePolicy', error);
		}
	});
}

export const createDevicePolicySaga = <T extends DeviceSettingsType>(
	type: T,
	responsibility: Responsibility,
	requiredCapability: ManagementCapability,
	checkAndSetSettings: (properties: IPolicyCheckAndSetProperties<T>) => AsyncGenerator<any, CheckAndSetResult>,
) =>
	whenResponsible(
		[Responsibility.POLICY, responsibility],
		whenCapable(
			requiredCapability,
			withDependencies(
				[
					'propertyStorage',
					'managementDriver',
					'proprietaryTimerStorage',
					'applicationVersion',
					'staticBaseUrl',
					'telemetryIntervals',
					'powerActionTimer',
				],
				function* ({
					propertyStorage,
					managementDriver,
					proprietaryTimerStorage: timerStorage,
					applicationVersion,
					staticBaseUrl,
					telemetryIntervals,
					powerActionTimer,
				}) {
					type LocalPolicyItemBase = IPolicyItem<T>;
					const powerActionRules = async () => await powerActionTimer.rules();
					const compareItemTypeToLocal = (policyItem: IPolicyItem) => policyItem.type === type;

					const periodMs = telemetryIntervals?.policy ?? 0;
					yield registerTelemetry('POLICY', periodMs, function* () {
						const failsCount = new FailsCount(type, propertyStorage);
						let policy: LocalPolicyItemBase | undefined;

						try {
							const devicePolicy: IPolicyItem[] = yield propertyStorage.getValueOrDefault<IPolicyItem[]>(Property.DEVICE_POLICY, []);
							debug('device policy %s enforcement', type);
							const fails: number = yield failsCount.get();

							if (!devicePolicy.some(compareItemTypeToLocal) || fails > POLICY_ACCEPTABLE_FAILS_COUNT) {
								return;
							}
							policy = devicePolicy.filter(compareItemTypeToLocal).pop() as LocalPolicyItemBase;
							debug('policy type %s item: %s', type, JSON.stringify(policy));

							const settingsChanged: CheckAndSetResult = yield asyncCall(checkAndSetSettings, {
								policy,
								managementDriver,
								propertyStorage,
								timerStorage,
								powerActionRules,
								getStaticBaseUrl: () => staticBaseUrl,
								applicationVersion,
							});

							debug('was policy %s set', type, settingsChanged);
							if (settingsChanged === CheckAndSetResult.CHANGED) {
								yield put<ApplyPolicySucceeded<typeof type>>({
									type: ApplyPolicySucceeded,
									name: type,
								});
							}

							if (settingsChanged === CheckAndSetResult.CHANGED) {
								yield failsCount.add();
							} else {
								yield failsCount.reset();
							}
						} catch (error) {
							yield failsCount.add();
							debug(`checking and setting device policy ${type} failed`, error);
						}

						const finalFails: number = yield failsCount.get();
						const failThresholdHit = finalFails > POLICY_ACCEPTABLE_FAILS_COUNT;
						if (failThresholdHit && typeof policy !== 'undefined') {
							yield put({
								type: ApplyPolicyFailed,
								name: type,
								data: policy,
							} as IAction<typeof ApplyPolicyFailed>);
						}
					});
				},
			),
		),
	);
