import { bindAndTakeEvery } from '../../../Socket/socketActionCreator';
import { call, put, fork } from 'redux-saga/effects';
import {
	UpgradeDeviceFirmwareVersion,
	UpgradeDeviceFirmwareVersionStarted,
	UpgradeDeviceFirmwareVersionFailed,
	UpgradeDeviceFirmwareVersionProgressed,
	NotifyDeviceFirmwareVersion,
	NotifyDeviceFirmwareType,
} from '@signageos/actions/dist/Device/Firmware/deviceFirmwareActions';
import { AuthenticationSucceed } from '@signageos/actions/dist/Authentication/authenticationActions';
import { createChannel, takeEvery as takeEveryChannelMessage, IChannel } from '../../../ReduxSaga/channels';
import ManagementCapability from '../../../NativeDevice/Management/ManagementCapability';
import { IResponsibilities } from '../../../Feature/Responsibilities';
import Responsibility from '../../../Feature/Responsibility';
import { whenCapable } from '../../../Feature/capable';
import { awaitDependencies, withDependencies } from '../../../DI/dependencyInjection';
import { SagaIterator } from 'redux-saga';
import processCallback from '../../../Util/processCallback';
import { SagaIteratorWithPromises } from '../../../ReduxSaga/extendedEffects';
import IManagementDriver from '../../../NativeDevice/Management/IManagementDriver';

export interface IProgressMessage {
	progress: number;
}

const deviceFirmwareVersionUpgradeStart = (progressChannel: IChannel<IProgressMessage>, getDeviceUid: () => Promise<string>) =>
	whenCapable(
		ManagementCapability.FIRMWARE_UPGRADE,
		withDependencies(
			['managementDriver', 'staticBaseUrl'],
			function* ({ managementDriver, staticBaseUrl }, action: UpgradeDeviceFirmwareVersion): SagaIterator {
				yield fork(deviceFirmwareVersionUpgradeProgress, action.uid, progressChannel);

				try {
					yield put({
						type: UpgradeDeviceFirmwareVersionStarted,
						uid: action.uid,
					} as UpgradeDeviceFirmwareVersionStarted);

					const finalizeCallback: () => Promise<void> = yield call(
						[managementDriver, managementDriver.firmwareUpgrade],
						staticBaseUrl,
						action.version,
						(progress: number) => progressChannel.put({ progress }),
					);
					yield call(checkVersion(getDeviceUid));
					yield call(processCallback, finalizeCallback);
				} catch (error) {
					console.error(error);
					yield put({
						type: UpgradeDeviceFirmwareVersionFailed,
						uid: action.uid,
					} as UpgradeDeviceFirmwareVersionFailed);
				}
			},
		),
	);

export function* deviceFirmwareNotify(responsibilities: IResponsibilities, getDeviceUid: () => Promise<string>) {
	if (!responsibilities.has(Responsibility.FIRMWARE)) {
		return;
	}

	yield bindAndTakeEvery(AuthenticationSucceed, checkVersion(getDeviceUid));
	yield bindAndTakeEvery(AuthenticationSucceed, notifyFirmwareType);
}

export function* deviceFirmwareUpgrade(responsibilities: IResponsibilities, getDeviceUid: () => Promise<string>) {
	if (!responsibilities.has(Responsibility.FIRMWARE)) {
		return;
	}

	const progressChannel = createChannel<IProgressMessage>();
	yield bindAndTakeEvery(UpgradeDeviceFirmwareVersion, function* (action: UpgradeDeviceFirmwareVersion) {
		yield fork(deviceFirmwareVersionUpgradeStart(progressChannel, getDeviceUid), action);
	});
}

function* deviceFirmwareVersionUpgradeProgress(uid: string, progressChannel: IChannel<IProgressMessage>) {
	yield takeEveryChannelMessage(progressChannel, function* ({ progress }: IProgressMessage) {
		try {
			yield put({
				type: UpgradeDeviceFirmwareVersionProgressed,
				uid,
				progress,
			} as UpgradeDeviceFirmwareVersionProgressed);
		} catch (error) {
			console.error('deviceFirmwareVersionUpgradeProgress', error);
		}
	});
}

const checkVersion = (getDeviceUid: () => Promise<string>) =>
	function* (): SagaIteratorWithPromises {
		try {
			const deviceUid: string = yield call(getDeviceUid);
			const { managementDriver }: { managementDriver: IManagementDriver } = yield awaitDependencies(['managementDriver']);
			const version: string = yield call([managementDriver, managementDriver.firmwareGetVersion]);
			yield put<NotifyDeviceFirmwareVersion>({
				type: NotifyDeviceFirmwareVersion,
				deviceUid,
				version,
			});
		} catch (error) {
			console.error('deviceFirmwareVersionNotify failed', error);
		}
	};

const notifyFirmwareType = function* (): SagaIteratorWithPromises {
	try {
		const { managementDriver } = yield awaitDependencies(['managementDriver']);
		const firmwareType: string | null = yield call([managementDriver, managementDriver.firmwareGetType]);
		if (firmwareType) {
			yield put<NotifyDeviceFirmwareType>({
				type: NotifyDeviceFirmwareType,
				firmwareType,
			});
		}
	} catch (error) {
		console.error('notifyFirmwareType failed', error);
	}
};
