import moment from 'moment-timezone';
import { DeviceSettingsType } from '@signageos/common-types/dist/Device/Settings/DeviceSettingsType';
import { CheckAndSetResult, createDevicePolicySaga } from '../Policy/devicePolicySagas';
import IPolicyCheckAndSetProperties from '../Policy/IPolicyCheckAndSetProperties';
import { getDateTimeSettings } from './deviceDateTimeTelemetrySagas';
import ManagementCapability from '../../../NativeDevice/Management/ManagementCapability';
import Responsibility from '../../../Feature/Responsibility';
import Debug from 'debug';
import processCallback from '../../../Util/processCallback';

const debug = Debug('@signageos/front-display:Management:Policy:DateTime');

export const deviceDateTimePolicySaga = createDevicePolicySaga(
	DeviceSettingsType.DATETIME,
	Responsibility.TIME,
	ManagementCapability.SET_TIMEZONE,
	checkAndSetDateTimeSettings,
);

export async function* checkAndSetDateTimeSettings(properties: IPolicyCheckAndSetProperties<DeviceSettingsType.DATETIME>) {
	const preferredSettings = properties.policy.value;
	const currentSettings = await getDateTimeSettings(properties);
	const newSettings = {
		ntpServer: preferredSettings.ntpServer ?? currentSettings.ntpServer,
		timezone: preferredSettings.timezone ?? currentSettings.timezone,
	};

	const offsetNewTimezone = newSettings.timezone === null ? null : moment().tz(newSettings.timezone).utcOffset();
	const offsetCurrentTimezone = currentSettings.timezone === null ? null : moment().tz(currentSettings.timezone).utcOffset();

	const formattedNewNtp = newSettings.ntpServer === null ? null : newSettings.ntpServer.toLowerCase();
	const formattedCurrentNtp = currentSettings.ntpServer === null ? null : currentSettings.ntpServer.toLowerCase();

	const isSettingsChanged = offsetNewTimezone !== offsetCurrentTimezone || formattedNewNtp !== formattedCurrentNtp;
	debug('Datetime comparsion timezone offset proposed %d with current %d', offsetNewTimezone, offsetCurrentTimezone);
	if (newSettings.timezone && isSettingsChanged) {
		let finalizeCallback: void | (() => Promise<void>);
		if (formattedNewNtp) {
			debug('time manger setAuto method will be called');
			finalizeCallback = await properties.managementDriver.timeManager.setAuto(formattedNewNtp, newSettings.timezone);
		} else {
			debug('time manager setManual method will be called');
			finalizeCallback = await properties.managementDriver.timeManager.setManual(new Date().valueOf(), newSettings.timezone);
		}
		processCallback(finalizeCallback);
		return CheckAndSetResult.CHANGED;
	}
	return CheckAndSetResult.NOT_CHANGED;
}
