import IManagementDriver from '../../../NativeDevice/Management/IManagementDriver';
import { TimeManagerFinalizeCallback } from '../../../NativeDevice/TimeManager/ITimeManager';

export interface ITimeWithTimezone {
	timestamp: number;
	timezone: string | null;
	/**
	 * Offset is POSIX based. A negative offset indicates a time zone east of the Prime Meridian.
	 * A positive offset indicates the time zone west of the Prime Meridian.
	 */
	timezoneOffset: number;
	ntpServer?: string;
	/**
	 * Some webos version can not set ntpServer, instead we can set ntpEnabled which set auto clock on display.
	 */
	ntpEnabled?: boolean;
}

export async function setNtpOrManualTime(
	nativeDriver: IManagementDriver,
	timestampMs: number,
	timezone: string,
	ntpServer?: string,
): Promise<TimeManagerFinalizeCallback> {
	if (ntpServer && (await nativeDriver.timeManager.canSetAuto())) {
		return await nativeDriver.timeManager.setAuto(ntpServer, timezone);
	} else if (await nativeDriver.timeManager.canSetManual()) {
		return await nativeDriver.timeManager.setManual(timestampMs, timezone);
	}
	throw new Error('Device is not able to set time & timezone');
}

export async function getCurrentTime(nativeDriver: IManagementDriver): Promise<ITimeWithTimezone> {
	let settings: ITimeWithTimezone = {
		timestamp: await nativeDriver.timeManager.getEpochMillis(),
		timezone: null,
		timezoneOffset: (await nativeDriver.timeManager.getZoneOffset()).toPosix(),
		ntpEnabled: await nativeDriver.timeManager.isAutoEnabled(),
	};
	if (await nativeDriver.timeManager.canGetZoneId()) {
		settings = {
			...settings,
			timezone: await nativeDriver.timeManager.getZoneId(),
		};
	}
	if (await nativeDriver.timeManager.canGetNtpServer()) {
		settings = {
			...settings,
			ntpServer: (await nativeDriver.timeManager.getNtpServer()) ?? undefined,
		};
	}
	return settings;
}
