import { bindAndTakeEvery, bindAndTakeEveryOnlyWhenPlatform } from '../../../Socket/socketActionCreator';
import { call, fork, put, take } from 'redux-saga/effects';
import {
	UpdateDeviceTimeSettings,
	UpdateDeviceTimeSettingsSucceed,
	UpdateDeviceTimeSettingsFailed,
} from '@signageos/actions/dist/Device/DateTime/deviceDateTimeActions';
import { AuthenticationSucceed } from '@signageos/actions/dist/Authentication/authenticationActions';
import ManagementCapability from '../../../NativeDevice/Management/ManagementCapability';
import { setNtpOrManualTime } from './dateTimeFacade';
import Responsibility from '../../../Feature/Responsibility';
import { notifyCurrentTime } from '../../../Device/DateTime/currentTimeSyncSagas';
import { withDependencies } from '../../../DI/dependencyInjection';
import { Saga, SagaIterator } from '@redux-saga/types';
import { whenResponsible } from '../../../Feature/responsible';
import { whenCapable } from '../../../Feature/capable';
import { TimeChangeDetector } from '../../../DateTime/timeChangeDetector';
import { createChannel, takeEvery as takeEveryChannelMessage } from '../../../ReduxSaga/channels';
import { StopApplication } from '../../../Application/applicationActions';
import processCallback from '../../../Util/processCallback';

const TIME_NOTIFY_INTERVAL_MS = 30e3;
const TIME_NOTIFY_THRESHOLD_MS = 60e3;

export const updateTimeSettingsSaga = whenResponsible(
	Responsibility.TIME,
	whenCapable(
		ManagementCapability.SET_TIME,
		withDependencies(['managementDriver', 'propertyStorage'], function* ({ managementDriver }): SagaIterator {
			yield bindAndTakeEveryOnlyWhenPlatform(UpdateDeviceTimeSettings, function* (action: UpdateDeviceTimeSettings) {
				try {
					const finalizeCallback: void | (() => Promise<void>) = yield call(
						setNtpOrManualTime,
						managementDriver,
						action.timestamp,
						action.timezone,
						action.ntpServer,
					);
					yield put({
						type: UpdateDeviceTimeSettingsSucceed,
						uid: action.uid,
					});
					yield fork(notifyCurrentTime);
					yield call(processCallback, finalizeCallback);
				} catch (error) {
					console.error('updateTimeSettingsSage failed', error);
					yield put({
						type: UpdateDeviceTimeSettingsFailed,
						uid: action.uid,
					});
				}
			});
		}),
	),
);

const updatingTimeWhenChanged = function* (): SagaIterator {
	yield call(takeEverySignificantTimeChange, function* () {
		yield call(notifyCurrentTime);
	});
};

export const startupDeviceDateTimeSettings = whenResponsible(Responsibility.TIME, function* () {
	yield bindAndTakeEvery(AuthenticationSucceed, notifyCurrentTime);
	yield fork(updatingTimeWhenChanged);
});

const takeEverySignificantTimeChange = withDependencies(
	['managementDriver', 'propertyStorage'],
	function* ({ managementDriver }, saga: Saga<[]>): SagaIterator {
		const timeChangeDetector = new TimeChangeDetector(managementDriver, TIME_NOTIFY_INTERVAL_MS, TIME_NOTIFY_THRESHOLD_MS);
		const changesChannel = createChannel<void>((thisPut) => timeChangeDetector.start(() => thisPut()));
		try {
			yield call(takeEveryChannelMessage, changesChannel, saga);
			yield take(StopApplication);
		} finally {
			timeChangeDetector.stop();
		}
	},
);
