import { EventEmitter } from 'events';
import IWifi, { IWifiConnectOptions, IWifiDevice, WifiEvent } from '../../NativeDevice/Hardware/IWifi';
import WifiDevice from './MockWifiDevice';
import { getIntegerInclusive } from './utils/IntegerGenerator';

const wifiDevices = [
	new WifiDevice('School', true, 100, 'uni12345', 'WPA2 Enterprise'),
	new WifiDevice('Work', true, 80, 'happy1234', 'WPA2 Enterprise'),
	new WifiDevice('BrownsWifi', false, 75),
	new WifiDevice('SmithsWifi', true, 60, 'unsecure9876', 'WEP'),
	new WifiDevice('JaneAndroid', true, 50, 'hihi5678', 'WPA2 Personal'),
	new WifiDevice("Chris's Phone", true, 40, 'hello5678', 'WPA'),
];

enum WIFI_HW_STATE {
	CLIENT = 0,
	AP = 1,
	DISABLED = 2,
}

export default class MockWifi implements IWifi {
	private networkDetail: WifiDevice | null;
	private wifiState: WIFI_HW_STATE;
	private numberOfWifiScans: number;
	private countryCode: string | null;
	private eventEmitter: EventEmitter;

	constructor() {
		this.wifiState = WIFI_HW_STATE.DISABLED;
		this.numberOfWifiScans = 0;
		this.countryCode = null;
		this.eventEmitter = new EventEmitter();
	}

	public async enableClient(): Promise<void> {
		this.wifiState = WIFI_HW_STATE.CLIENT;
		this.eventEmitter.emit(WifiEvent.CLIENT_ENABLED);
	}

	public async disable(): Promise<void> {
		this.wifiState = WIFI_HW_STATE.DISABLED;
		this.eventEmitter.emit(WifiEvent.DISABLED);
	}

	public async enableAP(_ssid: string, _passphrase: string): Promise<void> {
		this.wifiState = WIFI_HW_STATE.AP;
		this.eventEmitter.emit(WifiEvent.AP_ENABLED);
	}

	public async isClientEnabled(): Promise<boolean> {
		return this.wifiState === WIFI_HW_STATE.CLIENT;
	}

	public async isAPEnabled(): Promise<boolean> {
		return this.wifiState === WIFI_HW_STATE.AP;
	}

	public async connect(_ssid: string, _password?: string, _options?: IWifiConnectOptions): Promise<void> {
		//password has to be between 8 and 63 characters
		if (_password && _password.length < 8 && _password.length > 63) {
			throw new Error('connection failed');
		} else {
			//simulating actual reaching the network, can happen it won't be reachable
			const devices = this.getWifiDevices();
			const filteredDevices = devices.filter((device: WifiDevice) => device.ssid === _ssid);

			//in reality the wifi with better signal strength is chosen, I chose the first one
			if (filteredDevices.length > 0 && (await filteredDevices[0].connect(_password))) {
				this.networkDetail = new WifiDevice(_ssid, _password ? true : false, 100, _password);
				this.eventEmitter.emit(WifiEvent.CLIENT_CONNECTED);
			} else {
				this.eventEmitter.emit(WifiEvent.CLIENT_CONNECT_REJECTED);
				throw new Error('connection failed');
			}
		}
	}

	public async disconnect(): Promise<void> {
		this.networkDetail = null;
		this.eventEmitter.emit(WifiEvent.CLIENT_DISCONNECTED);
	}

	public async getConnectedTo(): Promise<IWifiDevice | null> {
		return this.networkDetail
			? ({
					ssid: this.networkDetail.ssid,
					encrypted: this.networkDetail.encrypted,
					strength: this.networkDetail.strength,
				} as IWifiDevice)
			: null;
	}

	public async setCountry(_countryCode: string): Promise<void> {
		this.countryCode = _countryCode;
	}

	public async getCountry(): Promise<string | null> {
		return this.countryCode;
	}

	public async scan(): Promise<IWifiDevice[]> {
		if (!(await this.isClientEnabled()) && !(await this.isAPEnabled())) {
			throw new Error('Wifi have to be enabled before scan can begin.');
		}
		return this.getWifiDevices().map(({ ssid, encrypted, strength }: WifiDevice) => ({ ssid, encrypted, strength }));
	}

	public on(event: WifiEvent, listener: (...args: any[]) => void): void {
		this.eventEmitter.on(event, listener);
	}

	public once(event: WifiEvent, listener: (...args: any[]) => void): void {
		this.eventEmitter.once(event, listener);
	}

	public removeListener(event: WifiEvent, listener: (...args: any[]) => void): void {
		this.eventEmitter.removeListener(event, listener);
	}

	public removeAllListeners(event?: WifiEvent): void {
		this.eventEmitter.removeAllListeners(event);
	}

	private getWifiDevices(): WifiDevice[] {
		const GEN_MIN = 100;
		const GEN_MAX = 500;

		let devices = [...wifiDevices];

		if (this.numberOfWifiScans < 1) {
			devices = devices.slice(0, 3);
		} else {
			for (let index = devices.length - 2; index < devices.length; index++) {
				devices[index].ssid = devices[index].ssid + getIntegerInclusive(GEN_MIN, GEN_MAX);
			}
		}

		this.numberOfWifiScans++;
		return devices;
	}
}
