import { Dispatch } from 'redux';
import { all, put } from 'redux-saga/effects';
import startApplicationSaga from '../Application/startApplicationSaga';
import {
	afterRegistrationDeviceUidAuthentication,
	autoDeviceAuthentication,
	fallbackDeviceUidAuthentication,
	storeDeviceSession,
} from '../Authentication/authenticationSagas';
import { dependencyInjection } from '../DI/dependenciesSaga';
import { debugAllActions } from '../Debug/debugSagas';
import { updatingConfiguration } from '../Device/Configuration/deviceConfigurationSagas';
import { deviceConnectSaga } from '../Device/Connect/deviceConnectSagas';
import { notifyDeviceStatus } from '../Device/Status/deviceAliveSaga';
import { syncProprietaryTimerSaga } from '../Device/Timer/deviceProprietaryTimerSagas';
import { deviceVerification } from '../Device/Verification/deviceVerificationSagas';
import { fallbackDeviceRegistration, notifyDeviceFrontDisplayVersion } from '../Device/deviceSagas';
import IConfig from '../Display/IConfig';
import { IResponsibilities } from '../Feature/Responsibilities';
import { FrontCacheDriver, normalizeFrontCacheDriver } from '../NativeDevice/Default/combinedDriver';
import { IFrontManagementDriver } from '../NativeDevice/Management/IManagementDriver';
import { offlineActionsSaga } from '../Offline/offlineActionsSagas';
import OfflineCache from '../OfflineCache/OfflineCache';
import { IPropertyStorage } from '../Property/propertyStorage';
import { healthChecking } from '../Socket/socketHealthCheckSagas';
import { socketPlatformCreate } from '../Socket/socketSagas';
import { IPolymorphicSynchronizer } from '../Synchronization/PolymorphicSynchronizer/IPolymorphicSynchronizer';
import { commonSystemLogsSaga } from '../SystemLogs/commonSystemLogsSagas';
import * as TestFramework from '../Test/TestFramework';
import { runTestsSaga } from '../Test/testSagas';
import { createProprietaryTimerManagerForFront } from '../Timer/timerResolverFactory';
import { IWebWorkerFactory } from '../WebWorker/masterWebWorkerFactory';
import { browserEventEmitting } from './Applet/Browser/appletBrowserSagas';
import { IBundledApplet, NotifyBundledApplet } from './Applet/BundledApplet/bundledAppletActions';
import { dispatchAppletCommandToApplet } from './Applet/Command/appletCommandDispatchSagas';
import { currentTimestampEmiting } from './Applet/DateTime/appletDateTimeSagas';
import { storageStatusChangeEmitting } from './Applet/FileSystem/appletFileSystemSagas';
import { inputKeysEmitting } from './Applet/Input/appletKeySagas';
import { wifiEventEmitting } from './Applet/Management/Wifi/appletWifiSagas';
import { proximitySensorStatusChangeEmitting } from './Applet/Sensors/appletProximitySensorSagas';
import { streamEventEmitting } from './Applet/Stream/appletStreamSagas';
import { syncEventEmitting } from './Applet/Sync/appletSyncSagas';
import { activeAppletRestore, restoreOnAppletChange, videoEventEmitting } from './Applet/Video/appletVideoSagas';
import { activeAppletBinary, updateActiveAppletBinaryFile } from './Applet/appletBinarySagas';
import {
	bindAppletIframe,
	bindAppletMessages,
	notifyAppletTelemetry,
	updateAppletDeviceAuthHash,
	updateAppletUniqueHash,
} from './Applet/appletSagas';
import { getDefaultAppletTimingDefinition } from './Applet/defaultApplet';
import {
	deleteAppletTimingsDefinition,
	loadAppletTimingsDefinition,
	saveAppletBinary,
	saveAppletTimingsDefinition,
} from './AppletTiming/appletTimingCacheSagas';
import { controlActiveAppletTiming } from './AppletTiming/appletTimingControllerSagas';
import { startupDeviceDateTimeSettings } from './Device/DateTime/currentTimeSyncSagas';
import { IWeinreContainer } from './Device/Debug/deviceDebugHelper';
import { devicePerformDebugSettings, readSetDeviceDebugSettings } from './Device/Debug/deviceDebugSettingsSagas';
import { deviceDeprovision } from './Device/Deprovision/deviceDeprovisionSagas';
import {
	deviceAppletDisablingRestoresActiveApplet,
	devicePerformPowerAction,
	devicePerformScheduledPowerAction,
} from './Device/Power/devicePowerActionSagas';
import { deviceSchedulePowerAction } from './Device/Power/deviceSchedulePowerActionSagas';
import { startupDeviceRemoteControlSettingsSaga } from './Device/RemoteControl/deviceRemoteControlSagas';
import { appletTimerChecking } from './Device/Timer/timerControllerSagas';
import { dispatchVerificationHash } from './Device/Verification/deviceVerificationSagas';
import { cacheDeviceAuthHash, startupDeviceAuthHash } from './Device/deviceAuthenticationSagas';
import {
	getDeviceLocationAndTags,
	notifyDeviceApplicationVersion,
	notifyDeviceSaga,
	notifyFrontCapabilities,
	updateDeviceLocation,
	updateOrganizationTags,
	updateTitle,
} from './Device/deviceSagas';
import { keySequenceSaga, keyUpSaga } from './Input/keySagas';
import { motionSequenceSaga } from './Input/motionSagas';
import { showOSDSaga } from './Input/osdSagas';
import { bindNumericKeyboardSaga } from './Input/softwareKeyboardSagas';
import { networkConnection } from './Network/networkSagas';
import { controlEnablingSaga, generateControlPinSaga, reportDevicePinChangedSaga } from './Security/controlEnablingSagas';
import { systemLogsSaga as frontSystemLogsSaga } from './SystemLogs/systemLogsSagas';
import CacheTest from './Tests/CacheTest';
import FileSystemTest from './Tests/FileSystemTest';
import HTML5Test from './Tests/HTML5Test';
import VideoTest from './Tests/VideoTest';
import WeinreTest from './Tests/WeinreTest';
import { touchSaga } from './Touch/touchSaga';
import { IFrontState } from './frontReducers';

export interface IFrontSagaOptions {
	publicKey: string;
	sessionIdKey: string;
	global: Window & IWeinreContainer;
	nativeDriver: FrontCacheDriver;
	managementDriver: IFrontManagementDriver;
	synchronizer: IPolymorphicSynchronizer;
	frontAppletPrefix: string;
	frontDisplayVersion: string;
	autoVerification: { organizationUid: string; deviceName?: string } | undefined;
	webWorkerFactory: IWebWorkerFactory;
	applicationVersion: string;
	bundledApplet: null | IBundledApplet;
	responsibilities: IResponsibilities;
	/** Additional tests that should run, apart from the standard suite that always runs */
	frontExtraTests?: Promise<TestFramework.Describe>[];
	shortAppletFilesUrl?: boolean;
	logOfflineActions?: boolean;
}

export function* frontSaga(
	defaultConfig: IConfig,
	{
		publicKey,
		sessionIdKey,
		global,
		nativeDriver,
		managementDriver,
		synchronizer,
		frontAppletPrefix,
		frontDisplayVersion,
		autoVerification,
		webWorkerFactory,
		applicationVersion,
		bundledApplet,
		responsibilities,
		frontExtraTests,
		shortAppletFilesUrl,
		logOfflineActions = true,
	}: IFrontSagaOptions,
	getState: () => IFrontState,
	propertyStorage: IPropertyStorage,
	dispatch: Dispatch<IFrontState>,
) {
	const { frontDriver, cacheDriver } = normalizeFrontCacheDriver(nativeDriver);

	const offlineCache = new OfflineCache(frontDriver.fileSystem);
	const getNativeDriver = () => nativeDriver;
	const getFrontDriver = () => frontDriver;
	const getManagementDriver = () => managementDriver;
	const getSynchronizer = () => synchronizer;
	//const dataStorage = createDataStorage(getFrontDriver);
	const defaultTiming = bundledApplet
		? getDefaultAppletTimingDefinition(
				bundledApplet.uid,
				bundledApplet.version,
				bundledApplet.frontAppletVersion,
				bundledApplet.checksum,
				bundledApplet.config,
			)
		: undefined;
	const defaultTimingData = bundledApplet
		? {
				appletUid: bundledApplet.uid,
				appletVersion: bundledApplet.version,
				appletBinaryFile: bundledApplet.binaryFile,
				appletFrontAppletJsFile: bundledApplet.frontAppletBinaryFile,
			}
		: undefined;
	const proprietaryTimerStorage = frontDriver.proprietaryTimerStorage;
	const proprietaryTimerManager = createProprietaryTimerManagerForFront(responsibilities, proprietaryTimerStorage);

	yield all([
		// create DI container as first
		dependencyInjection({
			applicationVersion,
			managementDriver,
			frontDriver,
			propertyStorage,
			responsibilities,
			proprietaryTimerStorage,
		}),
		// shared sagas
		socketPlatformCreate(getFrontDriver, 'front', () => managementDriver.timeManager.getEpochMillis(), defaultConfig), // Must be first saga
		healthChecking(),
		updatingConfiguration(getFrontDriver, getManagementDriver, propertyStorage, defaultConfig, publicKey, 'front'),
		autoDeviceAuthentication(sessionIdKey, getFrontDriver),
		storeDeviceSession(sessionIdKey, getFrontDriver),
		fallbackDeviceUidAuthentication(getFrontDriver),
		afterRegistrationDeviceUidAuthentication(getFrontDriver),
		fallbackDeviceRegistration(getFrontDriver, autoVerification),
		// front sagas
		startupDeviceDateTimeSettings(),
		notifyDeviceSaga(getManagementDriver),
		notifyDeviceFrontDisplayVersion(frontDisplayVersion),
		notifyDeviceApplicationVersion(getFrontDriver, applicationVersion),
		controlActiveAppletTiming(webWorkerFactory, defaultTiming ? [defaultTiming] : undefined),
		bindAppletMessages(
			frontAppletPrefix,
			global,
			getState,
			getNativeDriver,
			getSynchronizer,
			offlineCache,
			shortAppletFilesUrl ?? false,
			proprietaryTimerStorage,
		),
		bindAppletIframe(frontAppletPrefix, global, getState, frontDisplayVersion, getManagementDriver),
		updateAppletDeviceAuthHash(frontAppletPrefix, global, getState),
		updateAppletUniqueHash(),
		activeAppletRestore(getFrontDriver, getSynchronizer),
		restoreOnAppletChange(),
		currentTimestampEmiting(global, frontAppletPrefix, getState),
		browserEventEmitting(global, frontAppletPrefix, getFrontDriver, getState),
		videoEventEmitting(global, frontAppletPrefix, getState),
		streamEventEmitting(global, frontAppletPrefix, getState),
		wifiEventEmitting(global, frontAppletPrefix, getManagementDriver, getState),
		proximitySensorStatusChangeEmitting(global, frontAppletPrefix, getManagementDriver, getState),
		syncEventEmitting(global, frontAppletPrefix, getState, synchronizer),
		inputKeysEmitting(global, frontAppletPrefix, getState),
		storageStatusChangeEmitting(global, frontAppletPrefix, getFrontDriver, getState),
		startupDeviceAuthHash(propertyStorage),
		cacheDeviceAuthHash(propertyStorage),
		deviceVerification(getFrontDriver),
		dispatchVerificationHash(global),
		deviceDeprovision(getFrontDriver, getManagementDriver, responsibilities, sessionIdKey),
		deviceAppletDisablingRestoresActiveApplet(),
		deviceSchedulePowerAction(propertyStorage),
		devicePerformScheduledPowerAction(getNativeDriver, propertyStorage, offlineCache),
		appletTimerChecking(responsibilities, proprietaryTimerManager),
		startupDeviceRemoteControlSettingsSaga(getFrontDriver),
		keyUpSaga(getFrontDriver),
		keySequenceSaga(global),
		motionSequenceSaga(global),
		bindNumericKeyboardSaga(),
		controlEnablingSaga(),
		generateControlPinSaga(),
		reportDevicePinChangedSaga(),
		deviceConnectSaga(),
		showOSDSaga(getFrontDriver, propertyStorage),
		networkConnection(getState, getFrontDriver, getManagementDriver),
		touchSaga(frontAppletPrefix, getState, getFrontDriver, global),
		updateTitle(global),
		updateDeviceLocation(propertyStorage),
		updateOrganizationTags(propertyStorage),
		dispatchAppletCommandToApplet(frontAppletPrefix, global, getState, bundledApplet),
		devicePerformDebugSettings(global, getFrontDriver, propertyStorage, frontAppletPrefix, getState),
		readSetDeviceDebugSettings(global, getFrontDriver, propertyStorage, frontAppletPrefix, getState),
		devicePerformPowerAction(getNativeDriver, propertyStorage, offlineCache),
		syncProprietaryTimerSaga(),
		runTestsSaga([
			CacheTest(cacheDriver),
			FileSystemTest(frontDriver),
			HTML5Test(dispatch),
			VideoTest(frontDriver),
			WeinreTest(global),
			...(frontExtraTests ?? []),
		]),
		activeAppletBinary(getState, getFrontDriver),
		loadAppletTimingsDefinition(propertyStorage),
		saveAppletTimingsDefinition(propertyStorage),
		deleteAppletTimingsDefinition(propertyStorage, defaultTiming ? [defaultTiming] : undefined),
		saveAppletBinary(),
		updateActiveAppletBinaryFile(shortAppletFilesUrl ?? false, offlineCache, defaultTimingData),
		...(logOfflineActions ? [offlineActionsSaga('front.', getFrontDriver, getManagementDriver)] : []),
		commonSystemLogsSaga(),
		frontSystemLogsSaga(getManagementDriver),
		debugAllActions(),
		...(bundledApplet
			? [
					put<NotifyBundledApplet>({
						type: NotifyBundledApplet,
						bundledApplet: bundledApplet,
					}),
				]
			: []),
		startApplicationSaga(),
		notifyDeviceStatus(),
		notifyFrontCapabilities(propertyStorage, getFrontDriver),
		notifyAppletTelemetry(propertyStorage),
		getDeviceLocationAndTags(),
	]);
}
