import { bindAndTakeEvery } from '../../Socket/socketActionCreator';
import { fork, put, PutEffect, take, takeEvery } from 'redux-saga/effects';
import { HandleKeyUp, HandleKeySequence } from '@signageos/actions/dist/Input/keyActions';
import { AuthenticationSucceed } from '@signageos/actions/dist/Authentication/authenticationActions';
import { DevicePinChanged } from '@signageos/actions/dist/Device/Input/deviceInputActions';
import { DeviceRemoteControlSet } from '../../Device/RemoteControl/deviceRemoteControlActions';
import IFrontDriver from '../../NativeDevice/Front/IFrontDriver';
import { KeyCode } from '../../NativeDevice/Input/KeyCode';
import { getNumericCode } from '../Input/numericSequence';
import Responsibility from '../../Feature/Responsibility';
import { DEFAULT_OPEN_PIN_CODE } from './security';
import { StopApplication } from '../../Application/applicationActions';
import { buffers, channel, SagaIterator } from 'redux-saga';
import ISecurity from './ISecurity';
import { withDependencies } from '../../DI/dependencyInjection';
import { whenResponsible } from '../../Feature/responsible';
import processCallback, { ICallback } from '../../Util/processCallback';

// Currently this action is both reduced in front and sent to server.
// Ideally, we would split that and send to server based on responsibility and subscription type.
export const putDevicePinChanged = (pin: string): PutEffect<DevicePinChanged> =>
	put({
		type: DevicePinChanged,
		pin,
	});

// TODO remove this and add this option to OSD
// Enabling/disabling remote control requires a Responsibility, same as management.
export const controlEnablingSaga = whenResponsible(
	Responsibility.KIOSK,
	withDependencies(['frontDriver'], function* ({ frontDriver }): SagaIterator {
		const { security } = frontDriver;

		yield takeEvery(HandleKeySequence, function* (handleKeySequence: HandleKeySequence) {
			try {
				const characterSequence = getNumericCode(handleKeySequence.sequence);
				if (!characterSequence) {
					return;
				}

				const isPinOk: boolean = yield security.verifyPin(characterSequence);
				if (isPinOk) {
					yield remoteControlSetEnabled(frontDriver, true);
				}
			} catch (error) {
				console.error('controlEnablingSaga:keySequence', error);
			}
		});
		yield takeEvery(HandleKeyUp, function* (handleKeyUp: HandleKeyUp) {
			try {
				const isSecure: boolean = yield security.isSecure();
				if (!isSecure) {
					// Don't lock the device if there's no PIN.
					return;
				}

				if (handleKeyUp.keyCode === KeyCode[KeyCode.LOCK]) {
					yield remoteControlSetEnabled(frontDriver, false);
				}
			} catch (error) {
				console.error('controlEnablingSaga:keyUp', error);
			}
		});
	}),
);

export const generateControlPinSaga = whenResponsible(
	Responsibility.KIOSK,
	withDependencies(['frontDriver'], function* ({ frontDriver }): SagaIterator {
		const { security } = frontDriver;

		yield fork(function* () {
			try {
				yield ensurePin(security);
			} catch (error) {
				console.error('generateControlPinSaga:ensurePin', error);
			}
		});
		yield bindAndTakeEvery(AuthenticationSucceed, function* () {
			try {
				yield ensurePin(security);
				if (yield security.verifyPin(DEFAULT_OPEN_PIN_CODE)) {
					// Authenticated (connected) device will always create unique secret PIN code
					const pin = generateRandomPinCode();
					yield security.setPin(pin);
					yield putDevicePinChanged(pin);
				}
			} catch (error) {
				console.error('generateControlPinSaga', error);
			}
		});
	}),
);

export const reportDevicePinChangedSaga = withDependencies(['frontDriver'], function* ({ frontDriver }): SagaIterator {
	const { security } = frontDriver;

	const pinSetChannel = channel<object>(buffers.none());

	yield takeEvery(pinSetChannel, function* () {
		try {
			const pin: string = yield security.getPin();
			yield putDevicePinChanged(pin);
		} catch (error) {
			console.error('reportDevicePinChangedSaga:devicePinChanged', error);
		}
	});

	const listener = () => pinSetChannel.put({}); // Ping channel every time PIN changes.
	const removeListener = security.onPinSet(listener);
	try {
		pinSetChannel.put({}); // Extra ping on app start with initial state.

		take(StopApplication);
	} finally {
		removeListener();
	}
});

async function ensurePin(security: ISecurity) {
	const isSecure = await security.isSecure();
	if (!isSecure) {
		await security.setPin(DEFAULT_OPEN_PIN_CODE);
	}
}

function generateRandomPinCode(): string {
	return Math.floor(Math.random() * 1e4)
		.toString()
		.padStart(4, '0');
}

function* remoteControlSetEnabled(nativeDriver: IFrontDriver, enabled: boolean) {
	const finalizeCallback: ICallback = yield nativeDriver.remoteControlSetEnabled(enabled);
	processCallback(finalizeCallback);
	yield put({
		type: DeviceRemoteControlSet,
		enabled,
	});
}
