import { StartApplication } from '@signageos/actions/dist/Application/applicationActions';
import { put, takeEvery } from 'redux-saga/effects';
import IActionWithEmitter from '../../Application/IActionWithEmitter';
import { runPeriodicTaskSagaWhileOpen, runPeriodicTaskSagaWhilePlatform } from '../../Saga/periodicTaskSaga';
import moment from 'moment-timezone';
import { now } from '@signageos/lib/dist/DateTime/dateTimeFactory';
import { ChangeLicenseState, ViolationSeverity } from './licenseActions';
import { IPropertyStorage } from '../../Property/propertyStorage';
import Property from '../../Property/Property';
import { CheckSocketHealth, SocketConnected } from '../../Socket/socketActions';
import Debug from 'debug';
const debug = Debug('@signageos/front-display:License:licenseSaga');

const INITIAL_DELAY_MS = 120e3; // 120 seconds - to give a time to connect device to web socket for the first time
const PERIOD_MS = 30 * 60e3; // 30 minutes
const SOFT_EXPIRATION_DAYS = 8 * 7; // 8 weeks - this time is in Terms & Conditions of Open product
const HARD_EXPIRATION_DAYS = 16 * 7; // 16 weeks - double the time of T&C to prevent apocalypse
const RESPONSE_TIMEOUT_MS = 60e3;

/**
 * License is taking care of OPEN devices, which has to be connected to our cloud with connection for pinging.
 * Only for open devices, the device can be banned for any signageos operation and showing content.
 * The first soft expiration will show overlay with warning.
 * The second hard expiration will trigger content disabling.
 */
export function* licenseSaga(
	propertyStorage: IPropertyStorage,
	initialDelayMs: number = INITIAL_DELAY_MS,
	periodMs: number = PERIOD_MS,
	responseTimeoutMs: number = RESPONSE_TIMEOUT_MS,
) {
	let lastSuccessfulPingAt: Date = new Date(0);

	yield takeEvery([SocketConnected, CheckSocketHealth], function* (action: (SocketConnected | CheckSocketHealth) & IActionWithEmitter) {
		try {
			yield new Promise<void>((resolve: () => void, reject: (error: Error) => void) => {
				const timeoutHandler = setTimeout(() => reject(new Error('License check action delivery timed out')), responseTimeoutMs);
				action.__emitter.once('delivered', () => {
					clearTimeout(timeoutHandler);
					resolve();
				});
			});

			lastSuccessfulPingAt = new Date();
			yield propertyStorage.setValue(Property.LAST_SUCCESSFUL_PING_AT, lastSuccessfulPingAt.valueOf());
		} catch (error) {
			debug('licenseSaga failed', error);
		}
	});

	let lastSeverity: ViolationSeverity = ViolationSeverity.none;
	yield takeEvery(ChangeLicenseState, function* (action: ChangeLicenseState) {
		lastSeverity = action.severity;
	});
	function* adaptWarningState(severity: ViolationSeverity) {
		if (severity !== lastSeverity) {
			yield put<ChangeLicenseState>({
				type: ChangeLicenseState,
				severity,
			});
		}
	}

	yield takeEvery(StartApplication, function* () {
		lastSuccessfulPingAt = new Date(
			yield propertyStorage.getValueOrDefault(Property.LAST_SUCCESSFUL_PING_AT, lastSuccessfulPingAt.valueOf()),
		);
		yield runPeriodicTaskSagaWhilePlatform(initialDelayMs, periodMs, function* () {
			yield adaptWarningState(ViolationSeverity.none);
		});
		yield runPeriodicTaskSagaWhileOpen(initialDelayMs, periodMs, function* () {
			if (isExpired(HARD_EXPIRATION_DAYS, lastSuccessfulPingAt)) {
				yield adaptWarningState(ViolationSeverity.hard);
			} else if (isExpired(SOFT_EXPIRATION_DAYS, lastSuccessfulPingAt)) {
				yield adaptWarningState(ViolationSeverity.soft);
			} else {
				yield adaptWarningState(ViolationSeverity.none);
			}
		});
	});
}

function isExpired(timeoutDays: number, date: Date) {
	return moment(date).add(timeoutDays, 'days').isBefore(now());
}
