import { put, takeEvery } from 'redux-saga/effects';
import { HandleKeySequence, HandleKeyUp } from '@signageos/actions/dist/Input/keyActions';
import { takeEveryAndBindWhenPlatform } from '../../Socket/socketActionCreator';
import { getNumericCode } from './numericSequence';
import { HideOSD, ShowOSD, ShowOSDTime } from '../AppletTiming/appletTimingActions';
import { KeyCode } from '../../NativeDevice/Input/KeyCode';
import { UpdateActiveAppletBinary } from '@signageos/actions/dist/Applet/appletActions';
import { withDependencies } from '../../DI/dependencyInjection';
import IFrontDriver from '../../NativeDevice/Front/IFrontDriver';
import FrontCapability from '../../NativeDevice/Front/FrontCapability';
import { IFrontState } from '../frontReducers';
import wait from '@signageos/lib/es6/Timer/wait';

const OSD_TIME_LIMIT = 10e3;

export function* putOpenOSD(frontDriver: IFrontDriver, state: IFrontState, waitTime: number): Generator {
	// JS API can call opening OSD multiple times, so it can create internally multiple calls in saga to open OSD.
	// Those requests then can then result as failed, and it will block requests from “first” opened OSD.
	const currentTime = Date.now();
	if (currentTime - state.applet.lastOpenedOSDTime < OSD_TIME_LIMIT) {
		console.warn('OSD is opened too often, skipping.');
		return {};
	}
	yield put({ type: ShowOSDTime, time: currentTime } as ShowOSDTime);

	// This wait fixes the issue when the OSD is opened instantly after the applet is started.
	// On Brightsign it might break calling the JS API, and the OSD is then unresponsive,
	// for that reason we need to wait few seconds.
	yield wait(waitTime);
	yield put({
		type: ShowOSD,
		appletBinaryFile: {
			localUri: frontDriver.getOSDUri(),
		},
	} as ShowOSD);
	return {};
}

export const showOSDSaga = withDependencies(['frontDriver'], function* ({ frontDriver }) {
	const { security } = frontDriver;

	let isCurrentlyOsdShown = false;

	yield takeEvery(ShowOSD, function* (): IterableIterator<any> {
		isCurrentlyOsdShown = true;
	});
	yield takeEveryAndBindWhenPlatform(UpdateActiveAppletBinary, function* (): IterableIterator<any> {
		isCurrentlyOsdShown = false;
	});
	yield takeEvery(HandleKeySequence, function* (handleKeySequence: HandleKeySequence) {
		try {
			const sequence = handleKeySequence.sequence;
			const keySequenceWithoutLastKey = sequence.slice(0, sequence.length - 1);
			const lastKey = sequence[sequence.length - 1];
			const characterSequence = getNumericCode(keySequenceWithoutLastKey);
			const matchesPin: boolean = yield security.verifyPin(characterSequence);
			const frontOsdSupported: boolean = yield frontDriver.frontSupports(FrontCapability.FRONT_OSD);
			if (matchesPin && KeyCode[lastKey as keyof typeof KeyCode] === KeyCode.OK && frontOsdSupported) {
				yield showOSD(frontDriver);
			}
		} catch (error) {
			console.error('openOSDSaga', error);
		}
	});
	yield takeEvery(HandleKeyUp, function* (handleKeyUp: HandleKeyUp) {
		const keyCode = handleKeyUp.keyCode as keyof typeof KeyCode;
		if (isCurrentlyOsdShown && KeyCode[keyCode] === KeyCode.EXIT) {
			yield hideOSD();
		}
	});
});

function showOSD(frontDriver: IFrontDriver) {
	const osdUri = frontDriver.getOSDUri();
	return put({
		type: ShowOSD,
		appletBinaryFile: {
			localUri: osdUri,
		},
	} as ShowOSD);
}

function hideOSD() {
	return put({ type: HideOSD } as HideOSD);
}
