import RegisterFailReason from '@signageos/actions/dist/Device/RegisterFailReason';
import { DeviceRegistrationFailed, UpdateDeviceName, UpdateDeviceInfo } from '@signageos/actions/dist/Device/deviceActions';
import { UpdateSerialNumberSupport } from '../../Device/deviceActions';
import { UpdateLocation, ILocation } from '@signageos/actions/dist/Location/locationActions';
import { IOrganizationTag, UpdateOrganizationTags } from '@signageos/actions/dist/Organization/Tag/organizationTagActions';
import { SetDeviceDebugSettings } from '@signageos/actions/dist/Device/Debug/deviceDebugActions';
import { NotifyCurrentTime } from '@signageos/actions/dist/Device/DateTime/deviceDateTimeActions';
import { NotifyDeviceFrontDisplayVersion } from '@signageos/actions/dist/Device/FrontDisplay/deviceFrontDisplayActions';
import { NotifyDeviceApplicationVersion } from '@signageos/actions/dist/Device/Application/deviceApplicationActions';
import { DevicePinChanged } from '@signageos/actions/dist/Device/Input/deviceInputActions';
import { ReadDeviceDebugSettings } from './Debug/deviceDebugLocalActions';
import { NetworkStatusChanged } from '../Network/networkActions';
import { UpdateDisplayFirmwareVersion } from './deviceActions';

export interface IDeviceState {
	failReason: RegisterFailReason | null;
	applicationType: string | null;
	osVersion: string | null;
	model: string | null;
	deviceName: string | null;
	debugOn: boolean;
	ethernetMacAddress: string | null;
	wifiMacAddress: string | null;
	isSerialNumberSupported: boolean;
	serialNumber: string | null;
	pinCode: string | null;
	frontDisplayVersion: string | null;
	applicationVersion: string | null;
	timezone: string | null;
	timestamp: number | null;
	ipAddress: string | null;
	interfaceName: string | null;
	activeInterface: string | null;
	firmwareVersion: string | null;
	location: ILocation | null;
	organizationTags: IOrganizationTag[];
}

const initialState: IDeviceState = {
	failReason: null,
	applicationType: null,
	osVersion: null,
	model: null,
	deviceName: null,
	debugOn: false,
	ethernetMacAddress: null,
	wifiMacAddress: null,
	isSerialNumberSupported: false,
	serialNumber: null,
	pinCode: null,
	frontDisplayVersion: null,
	applicationVersion: null,
	timezone: null,
	timestamp: null,
	ipAddress: null,
	interfaceName: null,
	activeInterface: null,
	location: null,
	organizationTags: [],
	firmwareVersion: null,
};

export function device(
	state: IDeviceState = initialState,
	action:
		| DeviceRegistrationFailed
		| UpdateDeviceInfo
		| UpdateSerialNumberSupport
		| DevicePinChanged
		| UpdateDeviceName
		| NotifyCurrentTime
		| NotifyDeviceFrontDisplayVersion
		| NotifyDeviceApplicationVersion
		| SetDeviceDebugSettings
		| ReadDeviceDebugSettings
		| UpdateLocation
		| UpdateOrganizationTags
		| NetworkStatusChanged
		| UpdateDisplayFirmwareVersion,
) {
	switch (action.type) {
		case DeviceRegistrationFailed:
			return {
				...state,
				failReason: action.reason,
			};
		case UpdateDeviceInfo:
			return {
				...state,
				...action, // Merge. Don't explicitly define keys with undefined values. Luckily keys in action and state are the same.
			};
		case UpdateSerialNumberSupport:
			return {
				...state,
				isSerialNumberSupported: action.supported,
			};
		case DevicePinChanged:
			return {
				...state,
				pinCode: action.pin,
			};
		case UpdateDeviceName:
			return {
				...state,
				deviceName: action.deviceName,
			};
		case NotifyCurrentTime:
			return {
				...state,
				timezone: action.timezone,
				timestamp: action.timestamp,
			};
		case NotifyDeviceFrontDisplayVersion:
			return {
				...state,
				frontDisplayVersion: action.version,
			};
		case NotifyDeviceApplicationVersion:
			return {
				...state,
				applicationVersion: action.applicationVersion,
			};
		case UpdateLocation:
			return {
				...state,
				location: action.location,
			};
		case UpdateOrganizationTags:
			return {
				...state,
				organizationTags: action.tags,
			};
		case UpdateDisplayFirmwareVersion:
			return {
				...state,
				firmwareVersion: action.version,
			};
		case NetworkStatusChanged:
			const activeInterface = action.data.find((network) => network.localAddress !== undefined);
			if (!activeInterface) {
				return {
					...state,
					activeInterface: null,
					ipAddress: null,
					interfaceName: null,
				};
			}
			switch (activeInterface.type) {
				case 'ethernet':
					return {
						...state,
						activeInterface: activeInterface.name || null,
						ipAddress: activeInterface.localAddress || null,
						ethernetMacAddress: activeInterface.macAddress || null,
					};
				case 'wifi':
					return {
						...state,
						activeInterface: activeInterface.name || null,
						ipAddress: activeInterface.localAddress || null,
						wifiMacAddress: activeInterface.macAddress || null,
					};
				case 'vpn':
					return {
						...state,
						activeInterface: activeInterface.name || null,
						ipAddress: activeInterface.localAddress || null,
					};
				default:
					return state;
			}
		default:
			return state;
	}
}
