import {
	SubscribeActiveApplet,
	UpdateActiveAppletBinaryFile,
	UpdateActiveAppletUniqueHash,
} from '@signageos/actions/dist/Applet/appletActions';
import PowerActionType from '@signageos/actions/dist/Device/Power/PowerActionType';
import {
	PerformPowerAction,
	PerformPowerActionFailed,
	PerformPowerActionSucceed,
} from '@signageos/actions/dist/Device/Power/devicePowerActions';
import { generateUniqueHash } from '@signageos/lib/dist/Hash/generator';
import _ from 'lodash';
import { put, takeEvery } from 'redux-saga/effects';
import { FrontCacheDriver, normalizeFrontCacheDriver } from '../../../NativeDevice/Default/combinedDriver';
import OfflineCache from '../../../OfflineCache/OfflineCache';
import Property from '../../../Property/Property';
import { IPropertyStorage } from '../../../Property/propertyStorage';
import { takeEveryAndBindWhenPlatform } from '../../../Socket/socketActionCreator';
import { deliver } from '../../../Socket/socketActionDeliverHelper';
import { ActiveAppletRestore } from '../../Applet/activeAppletActions';
import { PerformScheduledPowerAction, TriggerAppletOff, TriggerAppletOn } from './devicePowerActionActions';

export function* devicePerformPowerAction(
	getNativeDriver: () => FrontCacheDriver,
	propertyStorage: IPropertyStorage,
	offlineCache: OfflineCache,
) {
	yield takeEveryAndBindWhenPlatform(PerformPowerAction, function* ({ powerType, id }: PerformPowerAction) {
		const { frontDriver } = normalizeFrontCacheDriver(getNativeDriver());
		const deviceUid: string = yield frontDriver.getDeviceUid();
		try {
			yield* deliver({
				type: PerformPowerActionSucceed,
				id,
				deviceUid,
			} as PerformPowerActionSucceed);
			const powerActionProcessed = yield* performPowerAction(getNativeDriver, propertyStorage, offlineCache, powerType);
			if (!powerActionProcessed) {
				return;
			}
		} catch (error) {
			console.error(error);
			yield put({
				type: PerformPowerActionFailed,
				id,
				deviceUid,
			} as PerformPowerActionFailed);
		}
	});
}

export function* devicePerformScheduledPowerAction(
	getNativeDriver: () => FrontCacheDriver,
	propertyStorage: IPropertyStorage,
	offlineCache: OfflineCache,
) {
	yield takeEvery(PerformScheduledPowerAction, function* ({ powerType }: PerformScheduledPowerAction) {
		try {
			yield performPowerAction(getNativeDriver, propertyStorage, offlineCache, powerType);
		} catch (error) {
			console.error('devicePerformScheduledPowerAction failed', error);
		}
	});
}

export function* deviceAppletDisablingRestoresActiveApplet() {
	yield takeEvery(TriggerAppletOn, function* () {
		try {
			yield put({ type: UpdateActiveAppletUniqueHash, hash: generateUniqueHash() } as UpdateActiveAppletUniqueHash);
		} catch (error) {
			console.error('displayPowerOn failed', error);
		}
	});
	yield takeEvery(TriggerAppletOff, function* () {
		try {
			yield put({
				type: ActiveAppletRestore,
			} as ActiveAppletRestore);
		} catch (error) {
			console.error('displayPowerOff failed', error);
		}
	});
}

function* performPowerAction(
	getNativeDriver: () => FrontCacheDriver,
	propertyStorage: IPropertyStorage,
	offlineCache: OfflineCache,
	powerType: PowerActionType,
) {
	const { frontDriver, cacheDriver } = normalizeFrontCacheDriver(getNativeDriver());
	const deviceUid: string = yield frontDriver.getDeviceUid();
	switch (powerType) {
		case PowerActionType.APPLET_RELOAD:
			const fileUids: string[] = yield offlineCache.listFileUids('');
			yield Promise.all(fileUids.map((fileUid: string) => offlineCache.deleteFileAndDeleteDirectoryIfEmpty(fileUid, true)));
			const contents: string[] = yield cacheDriver.cacheGetAll();
			const contentKeysToDelete = Object.keys(contents).filter((key: Property) => _.values(Property).indexOf(key) === -1);
			yield Promise.all(contentKeysToDelete.map((uid: string) => cacheDriver.cacheDelete(uid)));
			yield propertyStorage.removeValue(Property.APPLET_TIMINGS_UPDATE_DEFINITION);
			yield put({
				type: UpdateActiveAppletBinaryFile,
				appletBinaryFile: null,
				appletFrontAppletJsFile: null,
			} as UpdateActiveAppletBinaryFile);
			yield put({ type: UpdateActiveAppletUniqueHash, hash: generateUniqueHash() } as UpdateActiveAppletUniqueHash);
			yield put({ type: SubscribeActiveApplet, deviceUid } as SubscribeActiveApplet);
			yield put({ type: ActiveAppletRestore } as ActiveAppletRestore);
			return true;
		case PowerActionType.APPLET_REFRESH:
			yield put({ type: UpdateActiveAppletUniqueHash, hash: generateUniqueHash() } as UpdateActiveAppletUniqueHash);
			yield put({ type: SubscribeActiveApplet, deviceUid } as SubscribeActiveApplet);
			yield put({ type: ActiveAppletRestore } as ActiveAppletRestore);
		case PowerActionType.APPLET_ENABLE:
			yield put({ type: TriggerAppletOn });
			return true;
		case PowerActionType.APPLET_DISABLE:
			yield put({ type: TriggerAppletOff });
			return true;
		default:
			return false;
	}
}

export function* performAppletRefresh() {
	yield put({
		type: PerformPowerAction,
		powerType: PowerActionType.APPLET_RELOAD,
	});
}
