import { takeEveryAndBindWhenPlatform } from '../../../Socket/socketActionCreator';
import { put, takeEvery } from 'redux-saga/effects';
import {
	SetDeviceDebugSettings,
	SetDeviceDebugSettingsSucceeded,
	SetDeviceDebugSettingsFailed,
} from '@signageos/actions/dist/Device/Debug/deviceDebugActions';
import { ReadDeviceDebugSettings } from './deviceDebugLocalActions';
import { startWeinreDebug } from './deviceDebugHelper';
import IDriver from '../../../NativeDevice/Front/IFrontDriver';
import { IGlobalContainer } from '../../front';
import { IPropertyStorage } from '../../../Property/propertyStorage';
import Property from '../../../Property/Property';
import { StartApplication } from '@signageos/actions/dist/Application/applicationActions';
import { ActiveAppletReady } from '@signageos/actions/dist/Applet/activeAppletActions';
import { IFrontState } from '../../frontReducers';
import { sendMessageToActiveAppletIfExists } from '../../Applet/sendAppletMessage';
import { SagaIteratorWithPromises } from '../../../ReduxSaga/extendedEffects';

export function* devicePerformDebugSettings(
	global: IGlobalContainer,
	getNativeDriver: () => IDriver,
	propertyStorage: IPropertyStorage,
	messageTypePrefix: string,
	getState: () => IFrontState,
) {
	yield takeEveryAndBindWhenPlatform(
		SetDeviceDebugSettings,
		setDeviceDebugSettingsCallback,
		global,
		getNativeDriver,
		propertyStorage,
		messageTypePrefix,
		getState,
	);
}

export function* setDeviceDebugSettingsCallback(
	global: IGlobalContainer,
	getNativeDriver: () => IDriver,
	propertyStorage: IPropertyStorage,
	messageTypePrefix: string,
	getState: () => IFrontState,
	{ weinreEnabled, appletEnabled, uid }: SetDeviceDebugSettings,
): SagaIteratorWithPromises {
	try {
		// Back compatibility with old platform (appletEnabled was weinreEnabled in past)
		weinreEnabled = typeof weinreEnabled === 'undefined' ? !!appletEnabled : !!weinreEnabled;
		yield propertyStorage.setValue(Property.DEBUG_WEINRE_ENABLED, weinreEnabled);
		if (weinreEnabled) {
			startWeinreDebug(global);
		}
		yield* setDebugInAppletIfSomeActive(global, getNativeDriver, propertyStorage, messageTypePrefix, getState, () => weinreEnabled);

		yield put({
			type: SetDeviceDebugSettingsSucceeded,
			uid,
		} as SetDeviceDebugSettingsSucceeded);
	} catch (e) {
		yield put({
			type: SetDeviceDebugSettingsFailed,
			uid,
		} as SetDeviceDebugSettingsFailed);
	}
}

export function* readSetDeviceDebugSettings(
	global: IGlobalContainer,
	getNativeDriver: () => IDriver,
	propertyStorage: IPropertyStorage,
	messageTypePrefix: string,
	getState: () => IFrontState,
) {
	yield takeEvery(StartApplication, readSetDeviceDebugSettingsCallback, propertyStorage);
	yield takeEvery(
		ActiveAppletReady,
		setDebugInAppletIfSomeActive,
		global,
		getNativeDriver,
		propertyStorage,
		messageTypePrefix,
		getState,
		() => getState().debug.debugOn,
	);
}

export function* readSetDeviceDebugSettingsCallback(propertyStorage: IPropertyStorage) {
	const debugOnFromStorage: boolean = yield propertyStorage.getValueOrDefault(Property.DEBUG_WEINRE_ENABLED, false);

	yield put({
		type: ReadDeviceDebugSettings,
		debugOn: debugOnFromStorage,
	} as ReadDeviceDebugSettings);
}

function* setDebugInAppletIfSomeActive(
	global: IGlobalContainer,
	getNativeDriver: () => IDriver,
	propertyStorage: IPropertyStorage,
	messageTypePrefix: string,
	getState: () => IFrontState,
	isEnabled: () => boolean,
) {
	const url: string = yield propertyStorage.getValueOrDefault(Property.DEBUG_WEINRE_SERVER_URL, '');
	const duid: string = yield getNativeDriver().getDeviceUid();
	const activeAppletUid = getState().applet.activeAppletUid;
	if (activeAppletUid) {
		yield sendMessageToActiveAppletIfExists(global, getState, {
			type: messageTypePrefix + '.debug.configure',
			enabled: isEnabled(),
			url,
			id: `${duid}_${activeAppletUid}`,
		});
	}
}
