import { bindAndTakeEvery, bindAndTakeEveryOnlyWhenPlatform } from '../../Socket/socketActionCreator';
import { put, takeEvery } from 'redux-saga/effects';
import { StartApplication } from '@signageos/actions/dist/Application/applicationActions';
import {
	UpdateActiveAppletBinary,
	UpdateAppletTimingsDefinition,
	IAppletTimingDefinition,
	LastActiveAppletMissed,
	UpdateActiveAppletBinaryFile,
} from '@signageos/actions/dist/Applet/appletActions';
import TimingFinishEventType from './TimingFinishEventType';
import Property from '../../Property/Property';
import { IPropertyStorage } from '../../Property/propertyStorage';
import { ActiveAppletRestore } from '../Applet/activeAppletActions';

export function* loadAppletTimingsDefinition(propertyStorage: IPropertyStorage) {
	yield takeEvery(StartApplication, function* () {
		try {
			const updateAppletTimingsDefinition: UpdateAppletTimingsDefinition | null = yield propertyStorage.getValueOrDefault(
				Property.APPLET_TIMINGS_UPDATE_DEFINITION,
				null,
			);
			if (updateAppletTimingsDefinition) {
				yield put(updateAppletTimingsDefinition);
			}
		} catch (error) {
			console.error('loadAppletTimingsDefinition failed', error);
		}
	});
}

export function* saveAppletTimingsDefinition(propertyStorage: IPropertyStorage) {
	yield bindAndTakeEveryOnlyWhenPlatform(UpdateAppletTimingsDefinition, function* (action: UpdateAppletTimingsDefinition) {
		try {
			yield propertyStorage.setValue<UpdateAppletTimingsDefinition>(Property.APPLET_TIMINGS_UPDATE_DEFINITION, action);
		} catch (error) {
			console.error('saveAppletTimingsDefinition failed', error);
		}
	});
}

export function* deleteAppletTimingsDefinition(propertyStorage: IPropertyStorage, defaultDefinitions?: IAppletTimingDefinition[]) {
	const defaultDefinitionsOrNull = (): IAppletTimingDefinition[] | null => {
		if (defaultDefinitions && defaultDefinitions.length > 0) {
			return defaultDefinitions;
		}
		return null;
	};

	const definitions = defaultDefinitionsOrNull();
	yield bindAndTakeEvery(LastActiveAppletMissed, deleteTimingsDefinitionCallback, propertyStorage, definitions);
}

export function* deleteTimingsDefinitionCallback(propertyStorage: IPropertyStorage, timingDefinitions: IAppletTimingDefinition[] | null) {
	yield propertyStorage.removeValue(Property.APPLET_TIMINGS_UPDATE_DEFINITION);

	if (!timingDefinitions) {
		yield put({
			type: UpdateActiveAppletBinaryFile,
			appletBinaryFile: null,
			appletFrontAppletJsFile: null,
		} as UpdateActiveAppletBinaryFile);
		yield put({ type: ActiveAppletRestore } as ActiveAppletRestore);
	}
}

export function* saveAppletBinary() {
	yield bindAndTakeEveryOnlyWhenPlatform(UpdateActiveAppletBinary, function* (action: UpdateActiveAppletBinary) {
		try {
			// only UPDATE_ACTIVE_APPLET_BINARY dispatched from are stored as definition for BC
			if (typeof (action as any).__source !== 'undefined') {
				yield put({
					type: UpdateAppletTimingsDefinition,
					definitions: [
						{
							appletUid: action.activeAppletUid,
							configuration: action.activeConfiguration,
							timmingChecksum: action.activeTimingChecksum,
							appletVersion: action.activeAppletVersion,
							appletVersionPostfix: action.activeAppletVersionPostfix,
							isPackage: action.activeAppletIsPackage,
							frontAppletVersion: action.activeAppletFrontAppletVersion || action.activeAppletHugApiVersion, // deprecated will be deleted
							finishEvent: {
								type: TimingFinishEventType[TimingFinishEventType.DURATION],
								data: null,
							},
							position: 1,
							metadata: action.activeAppletMetadata,
						} as IAppletTimingDefinition,
					],
				} as UpdateAppletTimingsDefinition);
			}
		} catch (error) {
			console.error('saveAppletBinary failed', error);
		}
	});
}
