import IDriver from '../../../NativeDevice/Front/IFrontDriver';
import ISyncConnectMessage from './ISyncConnectMessage';
import ISyncCloseMessage from './ISyncCloseMessage';
import ISyncJoinGroupMessage from './ISyncJoinGroupMessage';
import ISyncWaitMessage from './ISyncWaitMessage';
import ISyncCancelWaitMessage from './ISyncCancelWaitMessage';
import ISyncRequestSetValueMessage from './ISyncBroadcastValueMessage';
import { IPolymorphicSynchronizer } from '../../../Synchronization/PolymorphicSynchronizer/IPolymorphicSynchronizer';
import InternalSynchronizerError from '../Error/InternalSynchronizerError';
import ErrorCodes from '../Error/ErrorCodes';
import { HandlerResult, IHandlerParams } from '../IHandler';
import ISyncLeaveGroupMessage from './ISyncLeaveGroupMessage';
import ISyncIsMasterMessage from './ISyncIsMasterMessage';
import { appletSyncEngineToSynchronizerSynchronizerType } from './appletSyncHandler.utils';

export function* handleSyncMessage(
	messageTypePrefix: string,
	data:
		| ISyncConnectMessage
		| ISyncCloseMessage
		| ISyncJoinGroupMessage
		| ISyncLeaveGroupMessage
		| ISyncWaitMessage
		| ISyncCancelWaitMessage
		| ISyncRequestSetValueMessage,
	_nativeDriver: IDriver,
	synchronizer: IPolymorphicSynchronizer,
	_appletUid: string,
	_timingChecksum: string,
): HandlerResult {
	switch (data.type) {
		case messageTypePrefix + '.sync.connect':
			return yield handleSyncConnect(data as ISyncConnectMessage, synchronizer);
		case messageTypePrefix + '.sync.close':
			return yield handleSyncClose(synchronizer);
		// historically this was called 'sync.init' so it has to stay for backwards compatibility
		case messageTypePrefix + '.sync.init':
			return yield handleSyncJoinGroup(data as ISyncJoinGroupMessage, synchronizer);
		case messageTypePrefix + '.sync.leave_group':
			return yield handleSyncLeaveGroup(data as ISyncLeaveGroupMessage, synchronizer);
		case messageTypePrefix + '.sync.wait':
			return yield handleSyncWait(data as ISyncWaitMessage, synchronizer);
		case messageTypePrefix + '.sync.cancel_wait':
			return yield handleSyncCancelWait(data as ISyncCancelWaitMessage, synchronizer);
		// historically this was called 'sync.request_set_value' so it has to stay that way for backwards compatibility
		case messageTypePrefix + '.sync.request_set_value':
			return yield handleBroadcastValue(data as ISyncRequestSetValueMessage, synchronizer);
		case messageTypePrefix + '.sync.is_master':
			return yield handleIsMaster(data as ISyncIsMasterMessage, synchronizer);
		default:
			return null;
	}
}

function* handleSyncConnect({ serverUri, engine }: ISyncConnectMessage, synchronizer: IPolymorphicSynchronizer) {
	try {
		const syncType = appletSyncEngineToSynchronizerSynchronizerType(engine);
		yield synchronizer.connect(serverUri, syncType);
	} catch (error) {
		throw new InternalSynchronizerError({
			kind: 'internalSynchronizerErrorWithOrigin',
			message: "Synchronizer couldn't connect to the server.",
			code: ErrorCodes.SYNCHRONIZER_CONNECT_ERROR,
			origin: 'connect()',
			originStack: error.stack,
			originMessage: error.message,
		});
	}
	return {};
}

function* handleSyncClose(synchronizer: IPolymorphicSynchronizer) {
	try {
		yield synchronizer.close();
	} catch (error) {
		throw new InternalSynchronizerError({
			kind: 'internalSynchronizerErrorWithOrigin',
			message: "The connection wasn't closed correctly.",
			code: ErrorCodes.SYNCHRONIZER_CLOSE_ERROR,
			origin: 'close()',
			originStack: error.stack,
			originMessage: error.message,
		});
	}
	return {};
}

function* handleSyncJoinGroup(data: ISyncJoinGroupMessage, synchronizer: IPolymorphicSynchronizer): HandlerResult {
	try {
		yield synchronizer.joinGroup({
			groupName: data.groupName,
			deviceIdentification: data.deviceIdentification,
		});
	} catch (error) {
		throw new InternalSynchronizerError({
			kind: 'internalSynchronizerErrorWithOrigin',
			message: 'The group initialization failed.',
			code: ErrorCodes.SYNCHRONIZER_JOIN_GROUP_ERROR,
			origin: 'joinGroup()',
			originStack: error.stack,
			originMessage: error.message,
		});
	}

	return {};
}

function* handleSyncLeaveGroup(data: ISyncLeaveGroupMessage, synchronizer: IPolymorphicSynchronizer): HandlerResult {
	try {
		yield synchronizer.leaveGroup(data.groupName);
	} catch (error) {
		throw new InternalSynchronizerError({
			kind: 'internalSynchronizerErrorWithOrigin',
			message: 'Failed to leave the group.',
			code: ErrorCodes.SYNCHRONIZER_LEAVE_GROUP_ERROR,
			origin: 'leaveGroup()',
			originStack: error.stack,
			originMessage: error.message,
		});
	}
}

function* handleSyncWait(data: ISyncWaitMessage, synchronizer: IPolymorphicSynchronizer): HandlerResult {
	try {
		const responseData = yield synchronizer.wait({
			groupName: data.groupName,
			data: data.data,
			timeoutMs: data.timeout,
		});
		return { responseData };
	} catch (error) {
		throw new InternalSynchronizerError({
			kind: 'internalSynchronizerErrorWithOrigin',
			message: 'Wait failed for unexpected reason.',
			code: ErrorCodes.SYNCHRONIZER_WAIT_ERROR,
			origin: `wait(${data.groupName}, ${data.data}, ${data.timeout})`,
			originStack: error.stack,
			originMessage: error.message,
		});
	}
}

function* handleSyncCancelWait(data: ISyncCancelWaitMessage, synchronizer: IPolymorphicSynchronizer): HandlerResult {
	try {
		yield synchronizer.cancelWait(data.groupName);
		return {};
	} catch (error) {
		throw new InternalSynchronizerError({
			kind: 'internalSynchronizerErrorWithOrigin',
			message: 'Cancel wait failed for unexpected reason.',
			code: ErrorCodes.SYNCHRONIZER_CANCEL_WAIT_ERROR,
			origin: `cancelWait(${data.groupName})`,
			originStack: error.stack,
			originMessage: error.message,
		});
	}
}

function* handleBroadcastValue(data: ISyncRequestSetValueMessage, synchronizer: IPolymorphicSynchronizer): HandlerResult {
	try {
		yield synchronizer.broadcastValue({
			groupName: data.groupName,
			key: data.key,
			value: data.value,
		});
		return {};
	} catch (error) {
		throw new InternalSynchronizerError({
			kind: 'internalSynchronizerErrorWithOrigin',
			message: "Synchronizer couldn't set the value.",
			code: ErrorCodes.SYNCHRONIZER_BROADCAST_VALUE_ERROR,
			origin: `setValue(${data.groupName}, ${data.key}, ${data.value})`,
			originStack: error.stack,
			originMessage: error.message,
		});
	}
}

function* handleIsMaster(data: ISyncIsMasterMessage, synchronizer: IPolymorphicSynchronizer): HandlerResult {
	try {
		const isMaster = yield synchronizer.isMaster(data.groupName);
		return { isMaster };
	} catch (error) {
		throw new InternalSynchronizerError({
			kind: 'internalSynchronizerErrorWithOrigin',
			message: "Synchronizer couldn't get the master status.",
			code: ErrorCodes.SYNCHRONIZER_IS_MASTER_ERROR,
			origin: `isMaster(${data.groupName})`,
			originStack: error.stack,
			originMessage: error.message,
		});
	}
}

export default function* syncHandler({
	messageTypePrefix,
	data,
	frontDriver,
	synchronizer,
	appletUid,
	timingChecksum,
}: IHandlerParams): HandlerResult {
	return yield handleSyncMessage(
		messageTypePrefix,
		data as ISyncJoinGroupMessage | ISyncWaitMessage,
		frontDriver,
		synchronizer,
		appletUid,
		timingChecksum,
	);
}
