import { put, spawn, cancel, call } from 'redux-saga/effects';
import { ActiveAppletStreamPlay, ActiveAppletStreamStop } from '@signageos/actions/dist/Applet/activeAppletActions';
import { createChannel, takeEvery as takeEveryChannelMessage } from '../../../ReduxSaga/channels';
import IDriver from '../../../NativeDevice/Front/IFrontDriver';
import IStreamEvent from '../../../Stream/IStreamEvent';
import {
	IStreamErrorEvent,
	IStreamConnectedEvent,
	IStreamDisconnectedEvent,
	IStreamClosedEvent,
	IStreamTracksChangedEvent,
} from '../../../Stream/streamEvents';
import InternalStreamError from '../Error/InternalStreamError';
import ErrorCodes from '../Error/ErrorCodes';
import { HandlerResult, IHandlerParams } from '../IHandler';
import { NoMoreAvailableVideosError } from '../../../NativeDevice/Error/videoErrors';
import IStream from '../../../Stream/IStream';
import {
	IStreamGetTracksMessage,
	IStreamMessage,
	IStreamPauseMessage,
	IStreamPlayMessage,
	IStreamPrepareMessage,
	IStreamResetTrackMessage,
	IStreamResumeMessage,
	IStreamSelectTrackMessage,
	IStreamStopMessage,
} from './appletStreamMessages';
import { StreamConnected, StreamDisconnected, StreamError, StreamTracksChanged } from '../../../Stream/streamActions';
import { ITrackInfo } from '../../../Stream/IStreamPlayer';

export function* handleStreamMessage(
	messageTypePrefix: string,
	data:
		| IStreamPrepareMessage
		| IStreamPlayMessage
		| IStreamStopMessage
		| IStreamPauseMessage
		| IStreamResumeMessage
		| IStreamGetTracksMessage
		| IStreamSelectTrackMessage
		| IStreamResetTrackMessage,
	nativeDriver: IDriver,
	appletUid: string,
	timingChecksum: string,
): HandlerResult {
	switch (data.type) {
		case messageTypePrefix + '.stream.prepare':
			return yield handleStreamPrepare(data as IStreamPrepareMessage, nativeDriver);
		case messageTypePrefix + '.stream.play':
			return yield handleStreamPlay(messageTypePrefix, data as IStreamPlayMessage, nativeDriver, appletUid, timingChecksum);
		case messageTypePrefix + '.stream.stop':
			return yield handleStreamStop(data as IStreamStopMessage, nativeDriver, appletUid, timingChecksum);
		case messageTypePrefix + '.stream.pause':
			return yield handleStreamPause(data as IStreamPauseMessage, nativeDriver);
		case messageTypePrefix + '.stream.resume':
			return yield handleStreamResume(data as IStreamResumeMessage, nativeDriver);
		case messageTypePrefix + '.stream.get_tracks':
			return yield handleStreamGetTracks(data as IStreamGetTracksMessage, nativeDriver);
		case messageTypePrefix + '.stream.select_track':
			return yield handleStreamSelectTrack(data as IStreamSelectTrackMessage, nativeDriver);
		case messageTypePrefix + '.stream.reset_track':
			return yield handleStreamResetTrack(data as IStreamResetTrackMessage, nativeDriver);
		default:
			return null;
	}
}

export function* handleStreamPrepare(data: IStreamPrepareMessage, nativeDriver: IDriver): HandlerResult {
	const streamProtocol = resolveProtocolType(data);
	if (typeof streamProtocol !== 'undefined') {
		data.options = { ...data.options, protocol: streamProtocol };
	}
	try {
		yield nativeDriver.stream.prepare(data.uri, data.x, data.y, data.width, data.height, data.options);
	} catch (error) {
		if (error instanceof NoMoreAvailableVideosError) {
			console.warn('Video prepare failed because there are no more available video players');
		} else {
			throw new InternalStreamError({
				kind: 'internalStreamErrorWithOrigin',
				message: "Couldn't prepare the stream.",
				code: ErrorCodes.STREAM_PREPARE_ERROR,
				origin:
					`prepare(${data.uri}, ${data.x}, ${data.y}, ${data.width}, ${data.height}` +
					`${typeof data.options?.protocol !== 'undefined' ? ', ' + data.options.protocol : ''})`,
				originStack: error.stack,
				originMessage: error.message,
			});
		}
	}

	return {};
}

export function* handleStreamPlay(
	messageTypePrefix: string,
	data: IStreamPlayMessage,
	nativeDriver: IDriver,
	appletUid: string,
	timingChecksum: string,
): HandlerResult {
	const streamProtocol = resolveProtocolType(data);
	if (typeof streamProtocol !== 'undefined') {
		data.options = { ...data.options, protocol: streamProtocol };
	}

	let stream: IStream;
	try {
		stream = (yield nativeDriver.stream.play(data.uri, data.x, data.y, data.width, data.height, data.options)) as any;
	} catch (error) {
		throw new InternalStreamError({
			kind: 'internalStreamErrorWithOrigin',
			message: "Couldn't play the stream.",
			code: ErrorCodes.STREAM_PLAY_ERROR,
			origin:
				`play(${data.uri}, ${data.x}, ${data.y}, ${data.width}, ${data.height}` +
				`${typeof data.options?.protocol !== 'undefined' ? ', ' + data.options.protocol : ''})`,
			originStack: error.stack,
			originMessage: error.message,
		});
	}

	const streamChannel = createChannel<IStreamEvent<'error' | 'connected' | 'disconnected' | 'tracks_changed'>>(
		(putEvent: (message: IStreamEvent<'error' | 'connected' | 'disconnected' | 'tracks_changed'>) => void) => {
			stream.once('error', (event: IStreamErrorEvent) => putEvent(event));
			stream.on('connected', (event: IStreamConnectedEvent) => putEvent(event));
			stream.on('disconnected', (event: IStreamDisconnectedEvent) => putEvent(event));
			stream.on('tracks_changed', (event: IStreamTracksChangedEvent) => putEvent(event));
		},
	);

	const streamCloseChannel = createChannel<IStreamEvent<'closed'>>((putEvent: (message: IStreamEvent<'closed'>) => void) => {
		stream.once('closed', (event: IStreamClosedEvent) => putEvent(event));
	});

	// spawn process that will take stream events and create messages for applet from them
	const eventsTask = yield spawn(
		takeEveryChannelMessage,
		streamChannel,
		function* (streamMessage: IStreamErrorEvent | IStreamConnectedEvent | IStreamDisconnectedEvent | IStreamTracksChangedEvent) {
			try {
				switch (streamMessage.type) {
					case 'error':
						yield put({
							type: StreamError,
							uri: streamMessage.uri,
							x: streamMessage.x,
							y: streamMessage.y,
							width: streamMessage.width,
							height: streamMessage.height,
							protocol: streamMessage.options?.protocol ?? streamMessage.protocol,
							errorMessage: streamMessage.errorMessage,
						} as StreamError);
						break;
					case 'connected':
						yield put({
							type: StreamConnected,
							uri: streamMessage.uri,
							x: streamMessage.x,
							y: streamMessage.y,
							width: streamMessage.width,
							height: streamMessage.height,
							protocol: streamMessage.options?.protocol ?? streamMessage.protocol,
						} as StreamConnected);
						break;
					case 'disconnected':
						yield put({
							type: StreamDisconnected,
							uri: streamMessage.uri,
							x: streamMessage.x,
							y: streamMessage.y,
							width: streamMessage.width,
							height: streamMessage.height,
							protocol: streamMessage.options?.protocol ?? streamMessage.protocol,
						} as StreamDisconnected);
						break;
					case 'tracks_changed':
						yield put({
							type: StreamTracksChanged,
							uri: streamMessage.uri,
							x: streamMessage.x,
							y: streamMessage.y,
							width: streamMessage.width,
							height: streamMessage.height,
							tracks: streamMessage.tracks,
						} as StreamTracksChanged);
						break;
					default:
				}
			} catch (error) {
				console.error(messageTypePrefix + '.stream.play event sending failed', error);
			}
		},
	);

	// spawn process that waits until a "closed" event is emitted, cancels eventsTask process and ends itself
	yield spawn(function* () {
		yield call(streamCloseChannel.take);
		yield cancel(yield eventsTask);
	});

	yield put<ActiveAppletStreamPlay>({ type: ActiveAppletStreamPlay, appletUid, timingChecksum, data });
	return {};
}

export function* handleStreamStop(
	data: IStreamStopMessage,
	nativeDriver: IDriver,
	appletUid: string,
	timingChecksum: string,
): HandlerResult {
	try {
		yield nativeDriver.stream.stop(data.uri, data.x, data.y, data.width, data.height);
	} catch (error) {
		throw new InternalStreamError({
			kind: 'internalStreamErrorWithOrigin',
			message: "Couldn't stop the stream.",
			code: ErrorCodes.STREAM_STOP_ERROR,
			origin: `stop(${data.uri}, ${data.x}, ${data.y}, ${data.width}, ${data.height})`,
			originStack: error.stack,
			originMessage: error.message,
		});
	}

	yield put<ActiveAppletStreamStop>({ type: ActiveAppletStreamStop, appletUid, timingChecksum, data });
	return {};
}

export function* handleStreamPause(data: IStreamStopMessage, nativeDriver: IDriver) {
	try {
		yield nativeDriver.stream.pause(data.uri, data.x, data.y, data.width, data.height);
	} catch (error) {
		throw new InternalStreamError({
			kind: 'internalStreamErrorWithOrigin',
			message: "Couldn't pause the stream.",
			code: ErrorCodes.STREAM_PAUSE_ERROR,
			origin: `pause(${data.uri}, ${data.x}, ${data.y}, ${data.width}, ${data.height})`,
			originStack: error.stack,
			originMessage: error.message,
		});
	}
}

// TODO Is the same extra handling as in handleStreamPlay needed here?
export function* handleStreamResume(data: IStreamStopMessage, nativeDriver: IDriver) {
	try {
		yield nativeDriver.stream.resume(data.uri, data.x, data.y, data.width, data.height);
	} catch (error) {
		throw new InternalStreamError({
			kind: 'internalStreamErrorWithOrigin',
			message: "Couldn't resume the stream.",
			code: ErrorCodes.STREAM_RESUME_ERROR,
			origin: `resume(${data.uri}, ${data.x}, ${data.y}, ${data.width}, ${data.height})`,
			originStack: error.stack,
			originMessage: error.message,
		});
	}
}

export function* handleStreamGetTracks(data: IStreamGetTracksMessage, nativeDriver: IDriver) {
	try {
		const tracks: ITrackInfo[] = yield nativeDriver.stream.getTracks(data.videoId);
		return { tracks };
	} catch (error) {
		throw new InternalStreamError({
			kind: 'internalStreamErrorWithOrigin',
			message: "Couldn't get stream track info.",
			code: ErrorCodes.STREAM_GET_TRACKS_ERROR,
			origin: `getTracks(${JSON.stringify(data.videoId)})`,
			originStack: error.stack,
			originMessage: error.message,
		});
	}
}

export function* handleStreamSelectTrack(data: IStreamSelectTrackMessage, nativeDriver: IDriver) {
	try {
		yield nativeDriver.stream.selectTrack(data.videoId, data.trackType, data.groupId, data.trackIndex);
	} catch (error) {
		throw new InternalStreamError({
			kind: 'internalStreamErrorWithOrigin',
			message: "Couldn't select stream track.",
			code: ErrorCodes.STREAM_SELECT_TRACK,
			origin: `selectTrack(${JSON.stringify(data.videoId)}, ${data.trackType}, ${data.groupId}, ${data.trackIndex})`,
			originStack: error.stack,
			originMessage: error.message,
		});
	}
}

export function* handleStreamResetTrack(data: IStreamResetTrackMessage, nativeDriver: IDriver) {
	try {
		yield nativeDriver.stream.resetTrack(data.videoId, data.trackType, data.groupId);
	} catch (error) {
		throw new InternalStreamError({
			kind: 'internalStreamErrorWithOrigin',
			message: "Couldn't reset stream track.",
			code: ErrorCodes.STREAM_RESET_TRACK,
			origin: `resetTrack(${JSON.stringify(data.videoId)}, ${data.trackType}, ${data.groupId})`,
			originStack: error.stack,
			originMessage: error.message,
		});
	}
}

export function resolveProtocolType(data: IStreamPrepareMessage): string | undefined {
	// Legacy format unused formats (until front-applet 6.x)
	if ((data as any).protocolType !== undefined) {
		return (data as any).protocolType!.toUpperCase();
	}
	if (data.protocol !== undefined) {
		return data.protocol!.toUpperCase();
	}
	// New format from (front-applet 6.x), used in prepare and play functions
	return data.options?.protocol;
}

export default function* streamHandler({ messageTypePrefix, data, frontDriver, appletUid, timingChecksum }: IHandlerParams): HandlerResult {
	return yield handleStreamMessage(messageTypePrefix, data as IStreamMessage, frontDriver, appletUid, timingChecksum);
}
