import IManagementDriver from '../../../../NativeDevice/Management/IManagementDriver';
import {
	IGetNetworkInfoMessage,
	IGetNetworkInterfacesMessage,
	ISetNetworkManualMessage,
	ISetNetworkDHCPMessage,
	ISetNetworkManualLegacyMessage,
	ISetNetworkDHCPLegacyMessage,
	IDisableNetworkInterfaceMessage,
} from './messageTypes';
import { HandlerResult, IHandlerParams } from '../../IHandler';
import { EmptyObject } from '../../../../Util/EmptyObject';
import { NetworkInterfaces, NetworkInterfacesTypes } from '@signageos/common-types/dist/Device/Network/NetworkInterface';

const PREFIX = 'management';

export function* handleManagementNetworkMessage(
	messageTypePrefix: string,
	data:
		| IGetNetworkInfoMessage
		| IGetNetworkInterfacesMessage
		| ISetNetworkManualLegacyMessage
		| ISetNetworkDHCPLegacyMessage
		| ISetNetworkManualMessage
		| ISetNetworkDHCPMessage
		| IDisableNetworkInterfaceMessage,
	managementDriver: IManagementDriver,
): HandlerResult {
	switch (data.type) {
		case `${messageTypePrefix}.${PREFIX}.get_network_info`:
			return yield handleGetNetworkInfo(managementDriver);

		case `${messageTypePrefix}.${PREFIX}.get_network_interfaces`:
			return yield handleGetNetworkInterfaces(managementDriver);

		// deprecated
		case `${messageTypePrefix}.${PREFIX}.set_network_manual`:
			return yield handleSetNetworkManualLegacy(data as ISetNetworkManualLegacyMessage, managementDriver);

		// deprecated
		case `${messageTypePrefix}.${PREFIX}.set_network_dhcp`:
			return yield handleSetNetworkDHCPLegacy(data as ISetNetworkDHCPLegacyMessage, managementDriver);

		case `${messageTypePrefix}.${PREFIX}.set_network_manual_v2`:
			return yield handleSetNetworkManual(data as ISetNetworkManualMessage, managementDriver);

		case `${messageTypePrefix}.${PREFIX}.set_network_dhcp_v2`:
			return yield handleSetNetworkDHCP(data as ISetNetworkDHCPMessage, managementDriver);

		case `${messageTypePrefix}.${PREFIX}.disable_network_interface`:
			return yield handleDisableNetworkInterface(data as IDisableNetworkInterfaceMessage, managementDriver);

		default:
			return null;
	}
}

/**
 * Old version of the interface that we need to keep returning to the applet for backwards compatibility
 *
 * TODO implement new interface in front-applet so we can get rid of this
 */
interface INetworkInfo {
	localAddress?: string;
	ethernetMacAddress?: string;
	wifiMacAddress?: string;
	activeInterface?: NetworkInterfacesTypes;
	gateway?: string;
	netmask?: string;
	dns?: string[];
	interfaceName?: string;
	wifiStrength?: number;
	wifiSsid?: string;
}

async function handleGetNetworkInterfaces(managementDriver: IManagementDriver): Promise<{ interfaces: NetworkInterfaces[] }> {
	const interfaces = await managementDriver.network.listInterfaces();
	return { interfaces };
}

async function handleGetNetworkInfo(managementDriver: IManagementDriver): Promise<{ networkInfo: INetworkInfo }> {
	const networkInterfaces = await managementDriver.network.listInterfaces();
	const result: INetworkInfo = {};
	let activeInterface: NetworkInterfaces | null = null;

	for (let networkInterface of networkInterfaces) {
		switch (networkInterface.type) {
			case 'ethernet':
				result.ethernetMacAddress = networkInterface.macAddress;
				if (!activeInterface && networkInterface.localAddress) {
					activeInterface = networkInterface;
				}
				break;
			case 'mobile':
				if (!activeInterface && networkInterface.localAddress) {
					activeInterface = networkInterface;
				}
				break;
			case 'wifi':
				result.wifiMacAddress = networkInterface.macAddress;
				if (!activeInterface && networkInterface.localAddress) {
					result.wifiStrength = networkInterface.wifiStrength;
					result.wifiSsid = networkInterface.wifiSsid;
					activeInterface = networkInterface;
				}
				break;
			default:
				break;
		}
	}

	if (activeInterface) {
		result.activeInterface = activeInterface.type;
		result.localAddress = activeInterface.localAddress;
		result.gateway = activeInterface.gateway;
		result.netmask = activeInterface.netmask;
		result.dns = activeInterface.dns;
		result.interfaceName = activeInterface.name;
	}

	return { networkInfo: result };
}

async function handleSetNetworkManual(data: ISetNetworkManualMessage, managementDriver: IManagementDriver): Promise<EmptyObject> {
	await managementDriver.network.setManual(data.interfaceName, data.options);
	return {};
}

async function handleSetNetworkDHCP(data: ISetNetworkDHCPMessage, managementDriver: IManagementDriver): Promise<EmptyObject> {
	await managementDriver.network.setDHCP(data.interfaceName);
	return {};
}

async function handleSetNetworkManualLegacy(
	data: ISetNetworkManualLegacyMessage,
	managementDriver: IManagementDriver,
): Promise<EmptyObject> {
	const ifaceName = await getNetworkInterfaceNameByType(data.options.interface, managementDriver);
	if (ifaceName) {
		await managementDriver.network.setManual(ifaceName, {
			localAddress: data.options.localAddress,
			netmask: data.options.netmask,
			gateway: data.options.gateway,
			dns: data.options.dns,
		});
		return {};
	} else {
		throw new Error('network interface not found');
	}
}

async function handleSetNetworkDHCPLegacy(data: ISetNetworkDHCPLegacyMessage, managementDriver: IManagementDriver): Promise<EmptyObject> {
	const ifaceName = await getNetworkInterfaceNameByType(data.networkInterface, managementDriver);
	if (ifaceName) {
		await managementDriver.network.setDHCP(ifaceName);
		return {};
	} else {
		throw new Error('network interface not found');
	}
}

async function handleDisableNetworkInterface(
	data: IDisableNetworkInterfaceMessage,
	managementDriver: IManagementDriver,
): Promise<EmptyObject> {
	await managementDriver.network.disableInterface(data.interfaceName);
	return {};
}

async function getNetworkInterfaceNameByType(networkInterface: NetworkInterfacesTypes, managementDriver: IManagementDriver) {
	const ifaces = await managementDriver.network.listInterfaces();
	for (const iface of ifaces) {
		if (iface.type === networkInterface) {
			return iface.name;
		}
	}
	return null;
}

export default function* managementHandler({ messageTypePrefix, data, managementDriver }: IHandlerParams): HandlerResult {
	return yield handleManagementNetworkMessage(
		messageTypePrefix,
		data as
			| IGetNetworkInfoMessage
			| IGetNetworkInterfacesMessage
			| ISetNetworkManualLegacyMessage
			| ISetNetworkDHCPLegacyMessage
			| ISetNetworkManualMessage
			| ISetNetworkDHCPMessage
			| IDisableNetworkInterfaceMessage,
		managementDriver,
	);
}
