import { spawn, call, cancel } from 'redux-saga/effects';
import { generateUniqueHash } from '@signageos/lib/dist/Hash/generator';
import { createChannel, takeEvery as takeEveryChannelMessage } from '../../../ReduxSaga/channels';
import IDriver from '../../../NativeDevice/Front/IFrontDriver';
import { ISerialPort, SerialPortEvent } from '../../../NativeDevice/Hardware/ISerial';
import InternalHardwareError from '../Error/InternalHardwareError';
import ErrorCodes from '../Error/ErrorCodes';
import { HandlerResult, IHandlerParams } from '../IHandler';
import { sendMessageToActiveAppletIfExists } from '../sendAppletMessage';
import { IFrontState } from '../../frontReducers';
import IHardwareOpenSerialPortMessage from './IHardwareOpenSerialPortMessage';
import IHardwareCloseSerialPortMessage from './IHardwareCloseSerialPortMessage';
import IHardwareWriteToSerialPortMessage from './IHardwareWriteToSerialPortMessage';
import { Task } from 'redux-saga';

export function* handleHardwareSerialPortMessage(
	messageTypePrefix: string,
	data: IHardwareOpenSerialPortMessage | IHardwareCloseSerialPortMessage | IHardwareWriteToSerialPortMessage,
	window: Window,
	nativeDriver: IDriver,
	getState: () => IFrontState,
): HandlerResult {
	switch (data.type) {
		case messageTypePrefix + '.hardware.serial_port.open':
			return yield openSerialPort(messageTypePrefix, window, nativeDriver, getState, data as IHardwareOpenSerialPortMessage);
		case messageTypePrefix + '.hardware.serial_port.close':
			return yield closeSerialPort(data as IHardwareCloseSerialPortMessage);
		case messageTypePrefix + '.hardware.serial_port.write':
			return yield writeToSerialPort(data as IHardwareWriteToSerialPortMessage);
		default:
			return null;
	}
}

/*
 * Open serial ports have to be persisted between applet messages because later messages reference serial port instances open
 * by earlier messages.
 * Because these message handler are functional and don't have a state, it's a fundamental conflict of interest.
 * A clean solution would be to pass some serial port store object from the top but I decided not to do it, since it would
 * change a lot of code for a very small gain.
 * For now, having this global variable is good enough. I'm aware that it's ugly. If it becomes a problem, we'll refactor it.
 */
const serialPorts: { [refid: string]: ISerialPort } = {};

function getDefaultSerialDevice(nativeDriver: IDriver) {
	switch (nativeDriver.getApplicationType().toLowerCase()) {
		case 'tizen':
			return 'PORT1';
		case 'brightsign':
			return '0';
		case 'windows':
			return 'COM3';
		case 'default':
		case 'linux':
			return '/dev/ttyUSB0';
		case 'android':
			return '/dev/ttyusb0';
		default:
			return null;
	}
}

function* openSerialPort(
	messageTypePrefix: string,
	window: Window,
	nativeDriver: IDriver,
	getState: () => IFrontState,
	data: IHardwareOpenSerialPortMessage,
) {
	let serialPort: ISerialPort;
	if (data.options.device === undefined) {
		const defaultDevice = getDefaultSerialDevice(nativeDriver);
		if (defaultDevice === null) {
			throw new InternalHardwareError({
				kind: 'internalHardwareError',
				message: 'Failed to get default serial port for device. Please specify manually the device in options object.',
				code: ErrorCodes.HARDWARE_DEFAULT_DEVICE_NOT_FOUND,
			});
		}
		data.options.device = defaultDevice;
	}
	try {
		serialPort = yield nativeDriver.hardware.serial.openPort(data.options);
	} catch (error) {
		throw new InternalHardwareError({
			kind: 'internalHardwareErrorWithOrigin',
			message: 'Failed to open serial port',
			code: ErrorCodes.HARDWARE_OPEN_SERIAL_PORT_ERROR,
			origin: `openSerialPort(${JSON.stringify(data.options)})`,
			originStack: error.stack,
			originMessage: error.message,
		});
	}

	const refid = generateUniqueHash(10);
	serialPorts[refid] = serialPort;
	yield sendSerialPortDataToActiveApplet(messageTypePrefix, window, getState, refid, serialPort);
	return { refid };
}

function* sendSerialPortDataToActiveApplet(
	messageTypePrefix: string,
	window: Window,
	getState: () => IFrontState,
	refid: string,
	serialPort: ISerialPort,
) {
	const serialPortDataChannel = createChannel((putData: (data: Uint8Array) => void) => {
		serialPort.on(SerialPortEvent.DATA, (data: Uint8Array) => {
			putData(data);
		});
	});

	const serialPortCloseChannel = createChannel((putClosed: (closed: true) => void) => {
		serialPort.on(SerialPortEvent.CLOSE, () => putClosed(true));
	});

	const dataTask: Task = yield spawn(takeEveryChannelMessage, serialPortDataChannel, function* (data: string | Buffer) {
		yield sendMessageToActiveAppletIfExists(window, getState, {
			type: messageTypePrefix + '.hardware.serial_port.data',
			refid,
			data,
		});
	});

	yield spawn(function* () {
		yield call(serialPortCloseChannel.take);
		yield cancel(dataTask);
		delete serialPorts[refid];
	});
}

async function closeSerialPort(data: IHardwareCloseSerialPortMessage) {
	const { refid } = data;
	const serialPort = serialPorts[refid];
	if (!serialPort) {
		throw new InternalHardwareError({
			kind: 'internalHardwareError',
			message: "Closing serial port that isn't open",
			code: ErrorCodes.HARDWARE_CLOSE_SERIAL_PORT_ERROR,
		});
	}
	try {
		await serialPort.close();
	} catch (error) {
		throw new InternalHardwareError({
			kind: 'internalHardwareErrorWithOrigin',
			message: 'Failed to close serial port',
			code: ErrorCodes.HARDWARE_CLOSE_SERIAL_PORT_ERROR,
			origin: `closeSerialPort(${refid})`,
			originStack: error.stack,
			originMessage: error.message,
		});
	}
	return {};
}

async function writeToSerialPort(data: IHardwareWriteToSerialPortMessage) {
	const { refid } = data;
	const serialPort = serialPorts[refid];
	if (!serialPort) {
		throw new InternalHardwareError({
			kind: 'internalHardwareError',
			message: "Writting to a serial port that isn't open",
			code: ErrorCodes.HARDWARE_WRITE_TO_SERIAL_PORT_ERROR,
		});
	}
	try {
		await serialPort.write(data.data);
	} catch (error) {
		throw new InternalHardwareError({
			kind: 'internalHardwareErrorWithOrigin',
			message: 'Failed to write to serial port',
			code: ErrorCodes.HARDWARE_WRITE_TO_SERIAL_PORT_ERROR,
			origin: `writeToSerialPort(${refid})`,
			originStack: error.stack,
			originMessage: error.message,
		});
	}
	return {};
}

export default function* hardwareSerialPortHandler({
	messageTypePrefix,
	data,
	frontDriver,
	window,
	getState,
}: Pick<IHandlerParams, 'messageTypePrefix' | 'data' | 'frontDriver' | 'window' | 'getState'>): HandlerResult {
	return yield handleHardwareSerialPortMessage(
		messageTypePrefix,
		data as IHardwareOpenSerialPortMessage | IHardwareCloseSerialPortMessage | IHardwareWriteToSerialPortMessage,
		window,
		frontDriver,
		getState,
	);
}
