import IFileSystem from '../../../NativeDevice/IFileSystem';
import IListStorageUnitsMessage from './IListStorageUnitsMessage';
import IListFilesMessage from './IListFilesMessage';
import IFileExistsMessage from './IFileExistsMessage';
import IGetFileMessage from './IGetFileMessage';
import IWriteFileMessage from './IWriteFileMessage';
import IReadFileMessage from './IReadFileMessage';
import IMoveFileMessage from './IMoveFileMessage';
import ICopyFileMessage from './ICopyFileMessage';
import IDeleteFileMessage from './IDeleteFileMessage';
import IDownloadFileMessage from './IDownloadFileMessage';
import IExtractFileMessage from './IExtractFileMessage';
import ILinkMessage from './ILinkMessage';
import IGetFileChecksumMessage from './IGetFileChecksumMessage';
import ICreateDirectoryMessage from './ICreateDirectoryMessage';
import IIsDirectoryMessage from './IIsDirectoryMessage';
import InternalFileSystemError from '../Error/InternalFileSystemError';
import ErrorCodes from '../Error/ErrorCodes';
import { HandlerResult, IHandlerParams } from '../IHandler';
import { NotSupportedMethodError } from '../../../NativeDevice/Error/basicErrors';
import IAppendFileMessage from './IAppendFileMessage';
import { IFilePath } from '../../../NativeDevice/fileSystem';
import ICreateArchiveFileMessage from './ICreateArchiveFileMessage';

export function* handleFileSystemMessage(
	messageTypePrefix: string,
	data:
		| IListStorageUnitsMessage
		| IListFilesMessage
		| IFileExistsMessage
		| IGetFileMessage
		| IWriteFileMessage
		| IReadFileMessage
		| ICopyFileMessage
		| IMoveFileMessage
		| IDeleteFileMessage
		| IDownloadFileMessage
		| IExtractFileMessage
		| ILinkMessage
		| IGetFileChecksumMessage
		| ICreateDirectoryMessage
		| IIsDirectoryMessage
		| ICreateArchiveFileMessage,
	fileSystem: IFileSystem,
): HandlerResult {
	switch (data.type) {
		case messageTypePrefix + '.file_system.list_storage_units':
			return yield handleListStorageUnits(fileSystem);
		case messageTypePrefix + '.file_system.list_files':
			return yield handleListFiles(fileSystem, data as IListFilesMessage);
		case messageTypePrefix + '.file_system.exists':
			return yield handleFileExists(fileSystem, data as IFileExistsMessage);
		case messageTypePrefix + '.file_system.get_file':
			return yield handleGetFile(fileSystem, data as IGetFileMessage);
		case messageTypePrefix + '.file_system.write_file':
			return yield handleWriteFile(fileSystem, data as IWriteFileMessage);
		case messageTypePrefix + '.file_system.append_file':
			return yield handleAppendFile(fileSystem, data as IAppendFileMessage);
		case messageTypePrefix + '.file_system.read_file':
			return yield handleReadFile(fileSystem, data as IReadFileMessage);
		case messageTypePrefix + '.file_system.copy_file':
			return yield handleCopyFile(fileSystem, data as ICopyFileMessage);
		case messageTypePrefix + '.file_system.move_file':
			return yield handleMoveFile(fileSystem, data as IMoveFileMessage);
		case messageTypePrefix + '.file_system.delete_file':
			return yield handleDeleteFile(fileSystem, data as IDeleteFileMessage);
		case messageTypePrefix + '.file_system.download_file':
			return yield handleDownloadFile(fileSystem, data as IDownloadFileMessage);
		case messageTypePrefix + '.file_system.extract_file':
			return yield handleExtractFile(fileSystem, data as IExtractFileMessage);
		case messageTypePrefix + '.file_system.link':
			return yield handleLink(fileSystem, data as ILinkMessage);
		case messageTypePrefix + '.file_system.get_file_checksum':
			return yield handleGetFileChecksum(fileSystem, data as IGetFileChecksumMessage);
		case messageTypePrefix + '.file_system.create_directory':
			return yield handleCreateDirectory(fileSystem, data as ICreateDirectoryMessage);
		case messageTypePrefix + '.file_system.is_directory':
			return yield handleIsDirectory(fileSystem, data as IIsDirectoryMessage);
		case messageTypePrefix + '.file_system.create_archive_file':
			return yield handleCreateArchive(fileSystem, data as ICreateArchiveFileMessage);
		case messageTypePrefix + '.file_system.wipeout':
			return yield handleWipeout(fileSystem);
		default:
			return null;
	}
}

function humanizeFilePath(filePath: IFilePath) {
	return `storageUnit=${filePath.storageUnit.type},filePath=${filePath.filePath}`;
}

async function handleListStorageUnits(fileSystem: IFileSystem) {
	try {
		const storageUnits = await fileSystem.listStorageUnits();
		return { storageUnits };
	} catch (error) {
		throw new InternalFileSystemError({
			kind: 'internalFileSystemErrorWithOrigin',
			message: 'Unexpected error occurred when listing storage units.',
			code: ErrorCodes.FILE_SYSTEM_LIST_STORAGE_UNITS_ERROR,
			origin: 'listStorageUnits()',
			originStack: error.stack,
			originMessage: error.message,
		});
	}
}

async function handleListFiles(fileSystem: IFileSystem, data: IListFilesMessage) {
	try {
		const files = await fileSystem.listFiles(data.directoryPath);
		return { files };
	} catch (error) {
		throw new InternalFileSystemError({
			kind: 'internalFileSystemErrorWithOrigin',
			message: 'Unexpected error occurred when listing files.',
			code: ErrorCodes.FILE_SYSTEM_LIST_FILES_ERROR,
			origin: `listFiles(${humanizeFilePath(data.directoryPath)})`,
			originStack: error.stack,
			originMessage: error.message,
		});
	}
}

async function handleFileExists(fileSystem: IFileSystem, data: IFileExistsMessage) {
	try {
		const exists = await fileSystem.exists(data.filePath);
		return { exists };
	} catch (error) {
		throw new InternalFileSystemError({
			kind: 'internalFileSystemErrorWithOrigin',
			message: 'Unexpected error occurred when checking for file existence.',
			code: ErrorCodes.FILE_SYSTEM_FILE_EXISTS_ERROR,
			origin: `exists(${humanizeFilePath(data.filePath)})`,
			originStack: error.stack,
			originMessage: error.message,
		});
	}
}

async function handleGetFile(fileSystem: IFileSystem, data: IGetFileMessage) {
	try {
		const file = await fileSystem.getFile(data.filePath);
		return { file };
	} catch (error) {
		throw new InternalFileSystemError({
			kind: 'internalFileSystemErrorWithOrigin',
			message: 'Unexpected error occurred when getting a file.',
			code: ErrorCodes.FILE_SYSTEM_GET_FILE_ERROR,
			origin: `getFile(${humanizeFilePath(data.filePath)})`,
			originStack: error.stack,
			originMessage: error.message,
		});
	}
}

async function handleWriteFile(fileSystem: IFileSystem, data: IWriteFileMessage) {
	try {
		await fileSystem.writeFile(data.filePath, data.contents);
		return {};
	} catch (error) {
		throw new InternalFileSystemError({
			kind: 'internalFileSystemErrorWithOrigin',
			message: 'Unexpected error occurred when writing a file.',
			code: ErrorCodes.FILE_SYSTEM_WRITE_FILE_ERROR,
			origin: `writeFile(${humanizeFilePath(data.filePath)}, ${data.contents.substr(0, 100)}...)`,
			originStack: error.stack,
			originMessage: error.message,
		});
	}
}

async function handleAppendFile(fileSystem: IFileSystem, data: IAppendFileMessage) {
	try {
		await fileSystem.appendFile(data.filePath, data.contents);
		return {};
	} catch (error) {
		throw new InternalFileSystemError({
			kind: 'internalFileSystemErrorWithOrigin',
			message: 'Unexpected error occurred when writing a file.',
			code: ErrorCodes.FILE_SYSTEM_APPEND_FILE_ERROR,
			origin: `appendFile(${humanizeFilePath(data.filePath)}, ${data.contents.substr(0, 100)}...)`,
			originStack: error.stack,
			originMessage: error.message,
		});
	}
}

async function handleReadFile(fileSystem: IFileSystem, data: IReadFileMessage) {
	try {
		const contents = await fileSystem.readFile(data.filePath);
		return { contents };
	} catch (error) {
		throw new InternalFileSystemError({
			kind: 'internalFileSystemErrorWithOrigin',
			message: 'Unexpected error occurred when reading a file.',
			code: ErrorCodes.FILE_SYSTEM_READ_FILE_ERROR,
			origin: `readFile(${humanizeFilePath(data.filePath)})`,
			originStack: error.stack,
			originMessage: error.message,
		});
	}
}

async function handleCopyFile(fileSystem: IFileSystem, data: ICopyFileMessage) {
	try {
		await fileSystem.copyFile(data.sourceFilePath, data.destinationFilePath, data.options);
		return {};
	} catch (error) {
		throw new InternalFileSystemError({
			kind: 'internalFileSystemErrorWithOrigin',
			message: 'Unexpected error occurred when copying a file.',
			code: ErrorCodes.FILE_SYSTEM_COPY_FILE_ERROR,
			origin:
				`copyFile(${humanizeFilePath(data.sourceFilePath)}, ${humanizeFilePath(data.destinationFilePath)}, ` +
				`{overwrite:${data.options?.overwrite}})`,
			originStack: error.stack,
			originMessage: error.message,
		});
	}
}

async function handleMoveFile(fileSystem: IFileSystem, data: IMoveFileMessage) {
	try {
		await fileSystem.moveFile(data.sourceFilePath, data.destinationFilePath, data.options);
		return {};
	} catch (error) {
		throw new InternalFileSystemError({
			kind: 'internalFileSystemErrorWithOrigin',
			message: 'Unexpected error occurred when moving a file.',
			code: ErrorCodes.FILE_SYSTEM_MOVE_FILE_ERROR,
			origin:
				`moveFile(${humanizeFilePath(data.sourceFilePath)}, ${humanizeFilePath(data.destinationFilePath)}), ` +
				`{overwrite:${data.options?.overwrite}})`,
			originStack: error.stack,
			originMessage: error.message,
		});
	}
}

async function handleDeleteFile(fileSystem: IFileSystem, data: IDeleteFileMessage) {
	try {
		await fileSystem.deleteFile(data.filePath, data.recursive);
		return {};
	} catch (error) {
		throw new InternalFileSystemError({
			kind: 'internalFileSystemErrorWithOrigin',
			message: 'Unexpected error occurred when deleting a file.',
			code: ErrorCodes.FILE_SYSTEM_DELETE_FILE_ERROR,
			origin: `deleteFile(${humanizeFilePath(data.filePath)}, ${data.recursive})`,
			originStack: error.stack,
			originMessage: error.message,
		});
	}
}

async function handleDownloadFile(fileSystem: IFileSystem, data: IDownloadFileMessage) {
	try {
		await fileSystem.downloadFile(data.filePath, data.sourceUri, data.headers);
		return {};
	} catch (error) {
		throw new InternalFileSystemError({
			kind: 'internalFileSystemErrorWithOrigin',
			message: 'Unexpected error occurred when downloading a file.',
			code: ErrorCodes.FILE_SYSTEM_DOWNLOAD_FILE_ERROR,
			origin: `downloadFile(${humanizeFilePath(data.filePath)}, ${data.sourceUri}, ${JSON.stringify(data.headers)})`,
			originStack: error.stack,
			originMessage: error.message,
		});
	}
}

async function handleExtractFile(fileSystem: IFileSystem, data: IExtractFileMessage) {
	try {
		await fileSystem.extractFile(data.archiveFilePath, data.destinationDirectoryPath, data.method);
		return {};
	} catch (error) {
		throw new InternalFileSystemError({
			kind: 'internalFileSystemErrorWithOrigin',
			message: 'Unexpected error occurred when extracting a file.',
			code: ErrorCodes.FILE_SYSTEM_EXTRACT_FILE_ERROR,
			origin:
				`extractFile(${humanizeFilePath(data.archiveFilePath)}, ${humanizeFilePath(data.destinationDirectoryPath)}, ` + `${data.method})`,
			originStack: error.stack,
			originMessage: error.message,
		});
	}
}

async function handleCreateArchive(fileSystem: IFileSystem, data: ICreateArchiveFileMessage) {
	try {
		await fileSystem.createArchive(data.archiveFilePath, data.archiveEntries);
		return {};
	} catch (error) {
		throw new InternalFileSystemError({
			kind: 'internalFileSystemErrorWithOrigin',
			message: 'Unexpected error occurred when creating an archive file.',
			code: ErrorCodes.FILE_SYSTEM_CREATE_ARCHIVE_FILE_ERROR,
			origin: `createArchive(${humanizeFilePath(data.archiveFilePath)}, ${JSON.stringify(data.archiveEntries)})`,
			originStack: error.stack,
			originMessage: error.message,
		});
	}
}

async function handleLink(fileSystem: IFileSystem, data: ILinkMessage) {
	try {
		await fileSystem.link(data.sourceFilePath, data.destinationFilePath);
		return {};
	} catch (error) {
		throw new InternalFileSystemError({
			kind: 'internalFileSystemErrorWithOrigin',
			message:
				error instanceof NotSupportedMethodError
					? 'Link is not supported on this platform.'
					: 'Unexpected error occurred when creating a link.',
			code: error instanceof NotSupportedMethodError ? ErrorCodes.FILE_SYSTEM_LINK_NOT_SUPPORTED : ErrorCodes.FILE_SYSTEM_LINK_ERROR,
			origin: `link(${humanizeFilePath(data.sourceFilePath)}, ${humanizeFilePath(data.destinationFilePath)})`,
			originStack: error.stack,
			originMessage: error.message,
		});
	}
}

async function handleGetFileChecksum(fileSystem: IFileSystem, data: IGetFileChecksumMessage) {
	try {
		const checksum = await fileSystem.getFileChecksum(data.filePath, data.hashType);
		return { checksum };
	} catch (error) {
		throw new InternalFileSystemError({
			kind: 'internalFileSystemErrorWithOrigin',
			message: 'Unexpected error occurred when getting file checksum.',
			code: ErrorCodes.FILE_SYSTEM_GET_CHECKSUM_ERROR,
			origin: `getFileChecksum(${humanizeFilePath(data.filePath)}, ${data.hashType})`,
			originStack: error.stack,
			originMessage: error.message,
		});
	}
}

async function handleCreateDirectory(fileSystem: IFileSystem, data: ICreateDirectoryMessage) {
	try {
		await fileSystem.createDirectory(data.directoryPath);
		return {};
	} catch (error) {
		throw new InternalFileSystemError({
			kind: 'internalFileSystemErrorWithOrigin',
			message: 'Unexpected error occurred when creating directory.',
			code: ErrorCodes.FILE_SYSTEM_CREATE_DIR_ERROR,
			origin: `createDirectory(${humanizeFilePath(data.directoryPath)})`,
			originStack: error.stack,
			originMessage: error.message,
		});
	}
}

async function handleIsDirectory(fileSystem: IFileSystem, data: IIsDirectoryMessage) {
	try {
		const isDirectory = await fileSystem.isDirectory(data.filePath);
		return { isDirectory };
	} catch (error) {
		throw new InternalFileSystemError({
			kind: 'internalFileSystemErrorWithOrigin',
			message: 'Unexpected error occurred when checking if a path is directory.',
			code: ErrorCodes.FILE_SYSTEM_IS_DIR_ERROR,
			origin: `isDirectory(${humanizeFilePath(data.filePath)})`,
			originStack: error.stack,
			originMessage: error.message,
		});
	}
}

async function handleWipeout(fileSystem: IFileSystem) {
	try {
		await fileSystem.wipeout();
		return {};
	} catch (error) {
		throw new InternalFileSystemError({
			kind: 'internalFileSystemErrorWithOrigin',
			message: 'Unexpected error occurred when wiping out the file system.',
			code: ErrorCodes.FILE_SYSTEM_WIPEOUT_ERROR,
			origin: 'wipeout()',
			originStack: error.stack,
			originMessage: error.message,
		});
	}
}

export default function* fileSystemHandler({ messageTypePrefix, data, frontDriver }: IHandlerParams): HandlerResult {
	return yield handleFileSystemMessage(
		messageTypePrefix,
		data as
			| IListStorageUnitsMessage
			| IListFilesMessage
			| IFileExistsMessage
			| IGetFileMessage
			| IWriteFileMessage
			| IReadFileMessage
			| ICopyFileMessage
			| IMoveFileMessage
			| IDeleteFileMessage
			| IDownloadFileMessage
			| IExtractFileMessage
			| IGetFileChecksumMessage
			| ICreateDirectoryMessage
			| IIsDirectoryMessage,
		frontDriver.fileSystem,
	);
}
