import path from 'path';
import * as url from 'url';
import createAppletFileNameHelpers from './appletFileNameHelpers';
import { IFile } from '../../NativeDevice/fileSystem';
import OfflineCache from '../../OfflineCache/OfflineCache';
import { synchronous } from '../../Lock/synchronousDecorator';
import { retriable } from '../../Util/retriableDecorator';

export class AppletBinaryDownloader {
	private appletFileNameHelpers: ReturnType<typeof createAppletFileNameHelpers>;

	constructor(
		shortAppletFilesUrl: boolean,
		private offlineCache: OfflineCache,
		private getStaticBaseUrl: () => string,
	) {
		this.appletFileNameHelpers = createAppletFileNameHelpers(shortAppletFilesUrl);
	}

	@synchronous()
	@retriable({ count: 10, initialTimeoutMs: 1e3, maxTimeoutMs: 5 * 60e3 })
	public async getOrDownloadAppletBinaries(
		appletUid: string,
		appletVersion: string,
		isPackage: boolean,
		frontAppletVersion?: string,
		appletVersionPostfix?: string,
		defaultTimingData?: {
			appletUid: string;
			appletVersion: string;
			appletBinaryFile: string;
			appletFrontAppletJsFile: string | undefined;
		},
	): Promise<{
		appletBinaryFile: IFile | null;
		frontAppletJsFile: IFile | null;
	}> {
		if (defaultTimingData?.appletUid === appletUid && defaultTimingData?.appletVersion === appletVersion) {
			return getDefaultAppletBinaries(defaultTimingData.appletBinaryFile, defaultTimingData.appletFrontAppletJsFile, window.location.href);
		} else {
			if (isPackage) {
				return await this.downloadAppletPackageToLocalStorage(appletUid, appletVersion, appletVersionPostfix);
			} else {
				return await this.downloadAppletBinariesToLocalStorage(appletUid, appletVersion, frontAppletVersion, appletVersionPostfix);
			}
		}
	}

	public async getOrDownloadFrontAppletJsFile(frontAppletVersion: string) {
		const frontAppletJsFileName = this.appletFileNameHelpers.getFrontAppletJsFileName(frontAppletVersion);
		const existsFrontAppletJsFile = await this.offlineCache.fileExists(frontAppletJsFileName);
		if (!existsFrontAppletJsFile) {
			const staticBaseUrl = this.getStaticBaseUrl(); // It has to be retrieved here to prevent error when loading from offline cache
			const frontAppletJsUri = staticBaseUrl + '/lib/front-applet/' + frontAppletVersion + '/bundle.js';
			await this.offlineCache.retriableDownloadFile(5, frontAppletJsFileName, frontAppletJsUri, undefined, true);
		}
		return await this.offlineCache.getFile(frontAppletJsFileName);
	}

	private async downloadAppletBinariesToLocalStorage(
		appletUid: string,
		appletVersion: string,
		frontAppletVersion?: string,
		appletVersionPostfix?: string,
	) {
		const appletBinaryFileName = this.appletFileNameHelpers.getAppletBinaryFileName(appletUid, appletVersion, appletVersionPostfix);
		const existsAppletBinaryFile = await this.offlineCache.fileExists(appletBinaryFileName);
		if (!existsAppletBinaryFile) {
			const staticBaseUrl = this.getStaticBaseUrl(); // It has to be retrieved here to prevent error when loading from offline cache
			const appletUri = this.appletFileNameHelpers.getAppletBinaryFileUri(staticBaseUrl, appletUid, appletVersion, appletVersionPostfix);
			try {
				await this.offlineCache.retriableDownloadFile(5, appletBinaryFileName, appletUri, undefined, true);
			} catch (error) {
				// TODO this only console logging of error can has negative effect and should be refactored
				console.error('Error during load last applet binary', error);
			}
		}

		return {
			frontAppletJsFile: frontAppletVersion ? await this.getOrDownloadFrontAppletJsFile(frontAppletVersion) : null,
			appletBinaryFile: await this.offlineCache.getFile(appletBinaryFileName),
		};
	}

	private async downloadAppletPackageToLocalStorage(appletUid: string, appletVersion: string, appletVersionPostfix?: string) {
		await this.ensurePackageIsDownloaded(appletUid, appletVersion, appletVersionPostfix);
		await this.ensurePackageIsExtracted(appletUid, appletVersion, appletVersionPostfix);
		const mainFilePath = await this.getPackageMainFilePath(appletUid, appletVersion, appletVersionPostfix);

		return {
			frontAppletJsFile: null, // multi-file package applets has front-applet bundled always
			appletBinaryFile: await this.offlineCache.getFile(mainFilePath),
		};
	}

	private async ensurePackageIsDownloaded(appletUid: string, appletVersion: string, appletVersionPostfix: string | undefined) {
		const packageFilePath = this.appletFileNameHelpers.getAppletPackageFilePath(appletUid, appletVersion, appletVersionPostfix);
		const existsPackageFile = await this.offlineCache.fileExists(packageFilePath);
		if (!existsPackageFile) {
			const staticBaseUrl = this.getStaticBaseUrl(); // It has to be retrieved here to prevent error when loading from offline cache
			const packageUri = this.appletFileNameHelpers.getAppletPackageFileUri(staticBaseUrl, appletUid, appletVersion, appletVersionPostfix);
			await this.offlineCache.retriableDownloadFile(5, packageFilePath, packageUri, undefined, true);
		}
	}

	private async ensurePackageIsExtracted(appletUid: string, appletVersion: string, appletVersionPostfix: string | undefined) {
		const configFilePath = this.appletFileNameHelpers.getAppletPackageConfigFilePath(appletUid, appletVersion, appletVersionPostfix);
		const packageFilePath = this.appletFileNameHelpers.getAppletPackageFilePath(appletUid, appletVersion, appletVersionPostfix);
		const packageDirectoryPath = this.appletFileNameHelpers.getAppletPackageDirectoryPath(appletUid, appletVersion, appletVersionPostfix);
		const existsConfigFile = await this.offlineCache.fileExists(configFilePath);
		if (!existsConfigFile) {
			await this.offlineCache.extractFile(packageFilePath, packageDirectoryPath, 'zip', true);
		}
	}

	private async getPackageMainFilePath(appletUid: string, appletVersion: string, appletVersionPostfix: string | undefined) {
		const configFilePath = this.appletFileNameHelpers.getAppletPackageConfigFilePath(appletUid, appletVersion, appletVersionPostfix);
		const configContent = await this.offlineCache.readFile(configFilePath);
		const config = JSON.parse(configContent);
		if (typeof config.main !== 'string') {
			throw new Error(
				`Main file of applet package is not specified: ${appletUid}, ${appletVersion}, ${appletVersionPostfix}, ${configContent}`,
			);
		}
		const packageDirectoryPath = this.appletFileNameHelpers.getAppletPackageDirectoryPath(appletUid, appletVersion, appletVersionPostfix);
		const mainFilePath = packageDirectoryPath + '/' + config.main;

		return mainFilePath;
	}
}

function resolveFrontAppletJsFile(frontAppletJsUriOrPath: string | undefined, defaultBaseUri: string) {
	if (typeof frontAppletJsUriOrPath === 'undefined') {
		return null;
	}

	if (isUrlLike(frontAppletJsUriOrPath)) {
		return { localUri: frontAppletJsUriOrPath };
	}

	frontAppletJsUriOrPath = getPathname(getAbsoluteUri(frontAppletJsUriOrPath, defaultBaseUri));

	return { localUri: getAbsoluteUri(frontAppletJsUriOrPath, defaultBaseUri) };
}

export function getDefaultAppletBinaries(
	appletBinaryUriOrPath: string,
	frontAppletJsUriOrPath: string | undefined,
	defaultBaseUri: string,
): {
	appletBinaryFile: IFile;
	frontAppletJsFile: IFile | null;
} {
	const appletBinaryUri = ensureAbsoluteUri(appletBinaryUriOrPath, defaultBaseUri);
	return {
		appletBinaryFile: { localUri: appletBinaryUri },
		frontAppletJsFile: resolveFrontAppletJsFile(frontAppletJsUriOrPath, defaultBaseUri),
	};
}

function ensureAbsoluteUri(uriOrPath: string, defaultBaseUri: string) {
	if (isUrlLike(uriOrPath)) {
		return uriOrPath;
	}

	return getAbsoluteUri(uriOrPath, defaultBaseUri);
}

function getPathname(uri: string) {
	return url.parse(uri).pathname!;
}

function getAbsoluteUri(filePath: string, baseUri: string) {
	const parsedBaseUri = url.parse(baseUri);
	if (!parsedBaseUri.pathname) {
		throw new Error(`Invalid applet binary file path`);
	}

	const pathname = getAbsolutePathname(filePath, parsedBaseUri.pathname);

	return url.format({
		...parsedBaseUri,
		pathname,
		search: undefined,
	});
}

function getAbsolutePathname(filePath: string, basePathname: string) {
	if (path.isAbsolute(filePath)) {
		return filePath;
	}

	const segmentedBasePathname = basePathname.split('/');
	segmentedBasePathname[segmentedBasePathname.length - 1] = filePath;
	const pathname = segmentedBasePathname.join('/');

	return pathname;
}

function isUrlLike(uriOrPath: string) {
	return url.parse(uriOrPath).protocol !== null;
}
