import { Volume } from 'memfs';
import { Volume as _Volume } from 'memfs/lib/volume';
import path from 'path';
import { EventEmitter } from 'events';
import { trimSlashesAndDots } from './fileSystemHelper';
import { IStorageUnit, IFilePath, IFile, IHeaders } from '../NativeDevice/fileSystem';
import HashAlgorithm from '../NativeDevice/HashAlgorithm';
import IFileSystem, { IArchiveInfo, ICopyFileOptions, IMoveFileOptions } from '../NativeDevice/IFileSystem';
import { Blob } from 'buffer';
import { getChecksumOfBlob } from '../Hash/checksum';

const capacity = 20 * 1024 * 1024; // 20 MBi
export const FAKE_INTERNAL_STORAGE_UNIT = {
	type: 'internal',
	capacity: capacity,
	freeSpace: capacity * 0.875,
	usableSpace: capacity * 0.875,
	removable: false,
};

export const EXISTING_FILE_NAME = 'storage/file.txt';
export const EXISTING_FILE_REMOTE_URI = 'https://example.com/file.txt';
export const EXISTING_FILE_CONTENT = 'some content';
export const EXISTING_FILE_CHECKSUM = {
	crc32: '431f313f',
	md5: '9893532233caff98cd083a116b013c0b',
};
export const ERRORS = {
	fileNotFound: 'File not found',
	pathIsNotDirectory: 'Path is not a directory',
	pathIsNotFile: 'Path is not a file',
};

/* c8 ignore start */
export async function mockFetch(sourceUri: string, _headers?: IHeaders) {
	if (sourceUri === EXISTING_FILE_REMOTE_URI) {
		return {
			status: 200,
			ok: true,
			text: () => Promise.resolve(EXISTING_FILE_CONTENT),
			blob: () => Promise.resolve(new Blob([EXISTING_FILE_CONTENT])),
		};
	} else {
		return {
			status: 404,
			ok: false,
			text: () => Promise.resolve(''),
			blob: () => Promise.resolve(new Blob([])),
		};
	}
}
/* c8 ignore stop */

export class StubMemoryFileSystem implements IFileSystem {
	private volume: _Volume = new Volume();
	private eventEmitter: EventEmitter = new EventEmitter();

	public async listStorageUnits(): Promise<IStorageUnit[]> {
		return [FAKE_INTERNAL_STORAGE_UNIT];
	}

	public onStorageUnitsChanged(_listener: () => void): void {
		// do nothing
	}

	public async initialize() {
		this.volume.mkdirpSync('/storage');
	}

	public async listFiles(directoryPath: IFilePath): Promise<IFilePath[]> {
		const exists = await this.exists(directoryPath);
		if (!exists) {
			throw new Error(ERRORS.fileNotFound);
		}

		const isDirectory = await this.isDirectory(directoryPath);
		if (!isDirectory) {
			throw new Error(ERRORS.pathIsNotDirectory);
		}

		const absolutePath = this.getAbsolutePath(directoryPath);
		const filenames = this.volume.readdirSync(absolutePath) as string[];
		return filenames.map(
			(filename: string) =>
				({
					storageUnit: directoryPath.storageUnit,
					filePath: trimSlashesAndDots(path.join(directoryPath.filePath, filename)),
				}) as IFilePath,
		);
	}

	public async getFile(filePath: IFilePath): Promise<(IFile & IFilePath) | null> {
		try {
			const fileMetadata = this.volume.lstatSync(this.getAbsolutePath(filePath));
			if (!fileMetadata || !fileMetadata.isFile()) {
				return null;
			}
			const blob = await this.readFileAsBlob(filePath);
			const file: IFile = {
				localUri: filePath.filePath,
				mimeType: blob.type,
				sizeBytes: blob.size,
				createdAt: fileMetadata.birthtimeMs,
				lastModifiedAt: fileMetadata.mtimeMs,
			};
			return {
				...file,
				...filePath,
			};
		} catch (error) {
			return null;
		}
	}

	public async readFile(filePath: IFilePath): Promise<string> {
		const blob = await this.readFileAsBlob(filePath);
		const contents = await blob.text();
		return contents;
	}

	public async readFileAsBlob(filePath: IFilePath): Promise<Blob> {
		const stats = this.volume.lstatSync(this.getAbsolutePath(filePath));
		if (!stats.isFile()) {
			throw new Error(ERRORS.pathIsNotFile);
		}
		const absolutePath = this.getAbsolutePath(filePath);
		const contents = this.volume.readFileSync(absolutePath);
		return new Blob([contents]);
	}

	public async writeFile(filePath: IFilePath, contents: string): Promise<void> {
		const fullPath = this.getAbsolutePath(filePath);
		this.volume.writeFileSync(fullPath, contents);
	}

	public async appendFile(filePath: IFilePath, contents: string): Promise<void> {
		const fullPath = this.getAbsolutePath(filePath);
		this.volume.appendFileSync(fullPath, contents);
	}

	public async exists(filePath: IFilePath): Promise<boolean> {
		const fullPath = this.getAbsolutePath(filePath);
		return this.volume.existsSync(fullPath);
	}

	public async downloadFile(filePath: IFilePath, sourceUri: string, headers?: IHeaders): Promise<void> {
		const response = await mockFetch(sourceUri, headers);
		if (!response.ok) {
			throw new Error(`Not OK response during downloadFile ${sourceUri}`);
		}
		const blob = await response.blob();
		await this.writeBlobAsFile(filePath, blob);
	}

	public async writeBlobAsFile(filePath: IFilePath, blob: Blob) {
		return new Promise<void>((resolve) => {
			const fullPath = this.getAbsolutePath(filePath);
			this.volume.writeFileSync(fullPath, blob.toString());
			resolve();
		});
	}

	public async uploadFile(filePath: IFilePath, uri: string, formKey: string, headers?: IHeaders): Promise<string> {
		let response = '';
		const responseCallback = (thisResponse: string) => {
			response = thisResponse;
		};
		this.eventEmitter.emit('upload', filePath, uri, formKey, headers, responseCallback);
		return response;
	}

	public onUpload(
		listener: (
			filePath: IFilePath,
			uri: string,
			formKey: string,
			headers: { [key: string]: string } | undefined,
			responseCallback: (response: string) => void,
		) => void,
	) {
		this.eventEmitter.on('upload', listener);
	}

	public async deleteFile(filePath: IFilePath, recursive: boolean): Promise<void> {
		if (!(await this.exists(filePath))) {
			if (recursive) {
				return;
			} else {
				throw new Error(ERRORS.fileNotFound);
			}
		}
		const fullPath = this.getAbsolutePath(filePath);
		if (await this.isDirectory(filePath)) {
			if (recursive) {
				await this.deleteFileRecursive(filePath);
			} else {
				this.volume.rmdirSync(fullPath);
			}
		} else {
			this.volume.unlinkSync(fullPath);
		}
	}

	public async copyFile(sourceFilePath: IFilePath, destinationFilePath: IFilePath, options?: ICopyFileOptions): Promise<void> {
		const sourceFullPath = this.getAbsolutePath(sourceFilePath);
		const destinationFullPath = this.getAbsolutePath(destinationFilePath);
		if (this.volume.existsSync(destinationFullPath)) {
			if (options?.overwrite) {
				this.volume.unlinkSync(destinationFullPath);
			} else {
				throw new Error('Destination file already exists');
			}
		}
		this.volume.copyFileSync(sourceFullPath, destinationFullPath);
	}

	public async moveFile(sourceFilePath: IFilePath, destinationFilePath: IFilePath, options?: IMoveFileOptions): Promise<void> {
		const sourceFullPath = this.getAbsolutePath(sourceFilePath);
		const destinationFullPath = this.getAbsolutePath(destinationFilePath);
		if (this.volume.existsSync(destinationFullPath)) {
			if (options?.overwrite) {
				this.volume.unlinkSync(destinationFullPath);
			} else {
				throw new Error('Destination file already exists');
			}
		}
		this.volume.renameSync(sourceFullPath, destinationFullPath);
	}

	public async link(sourceFilePath: IFilePath, destinationFilePath: IFilePath): Promise<void> {
		const sourceFullPath = this.getAbsolutePath(sourceFilePath);
		const destinationFullPath = this.getAbsolutePath(destinationFilePath);
		this.volume.linkSync(sourceFullPath, destinationFullPath);
	}

	public async getFileChecksum(filePath: IFilePath, hashType: HashAlgorithm): Promise<string> {
		const blob = await this.readFileAsBlob(filePath);
		return getChecksumOfBlob(blob, hashType);
	}

	public async extractFile(_archiveFilePath: IFilePath, _destinationDirectoryPath: IFilePath, _method: string): Promise<void> {
		// TODO implement when needed
		throw new Error('Method not implemented.');
	}

	public async createArchive(_archiveFilePath: IFilePath, _archiveEntries: IFilePath[]): Promise<void> {
		throw new Error('Method not implemented.');
	}

	public async getArchiveInfo(_archiveFilePath: IFilePath): Promise<IArchiveInfo> {
		throw new Error('Method not implemented.');
	}

	public async wipeout(): Promise<void> {
		return Promise.resolve();
	}

	public async createDirectory(directoryPath: IFilePath): Promise<void> {
		const fullPath = this.getAbsolutePath(directoryPath);
		this.volume.mkdirSync(fullPath);
	}

	public async isDirectory(filePath: IFilePath): Promise<boolean> {
		if (!(await this.exists(filePath))) {
			return false;
		}
		const fullPath = this.getAbsolutePath(filePath);
		return this.volume.lstatSync(fullPath).isDirectory();
	}

	private async deleteFileRecursive(filePath: IFilePath) {
		const fullPath = this.getAbsolutePath(filePath);

		if (await this.isDirectory(filePath)) {
			const subFiles = await this.listFiles(filePath);
			for (const subFile of subFiles) {
				await this.deleteFileRecursive(subFile);
			}
			this.volume.rmdirSync(fullPath);
		} else {
			this.volume.unlinkSync(fullPath);
		}
	}

	private getAbsolutePath(relativePath: IFilePath) {
		return '/' + relativePath.filePath;
	}
}
