import { posix as path } from 'path';
import _ from 'lodash';
import IFrontFileSystem, { IArchiveInfo, ICopyFileOptions, IMoveFileOptions } from '../NativeDevice/IFileSystem';
import { IFilePath, IFile, IStorageUnit, IHeaders, IMetadata, ISimpleResponse, ISimpleHeaders } from '../NativeDevice/fileSystem';
import { getChecksumOfFile, checksumString } from '../Hash/checksum';
import HashAlgorithm from '../NativeDevice/HashAlgorithm';
import { extractZipFile, getZipExtractSize } from '../Archive/extractor';
import { trimSlashesAndDots } from './fileSystemHelper';
import { NotSupportedMethodError } from '../NativeDevice/Error/basicErrors';
import Debug from 'debug';

const debug = Debug('@signageos/front-display:FileSystem:ProprietaryFileSystem');

const METADATA_DIRECTORY = '.metadata';

export default class ProprietaryFileSystem implements IFrontFileSystem {
	private fileSystem: FileSystem;
	private storageUnit: IStorageUnit;

	constructor(
		private window: Window,
		private fileSystemType: number = window.TEMPORARY,
		private storageSizeBytes: number = 100 * 1024 * 1024,
	) {}

	public async initialize() {
		try {
			await this.requestFileSystem(this.fileSystemType, this.storageSizeBytes);
		} catch (error) {
			console.warn(`Unable to request PERSISTENT file system. Use TEMPORARY instead`, error);
			this.fileSystemType = this.window.TEMPORARY;
			await this.requestFileSystem(this.fileSystemType, this.storageSizeBytes);
		}
		await this.listStorageUnits();
	}

	public async listFiles(directoryPath: IFilePath): Promise<IFilePath[]> {
		const fs = await this.getFileSystem(directoryPath.storageUnit);
		const directoryEntry = await this.getDirectoryEntry(fs, directoryPath.filePath);
		const directoryReader = directoryEntry.createReader();
		const entries = await this.readDirectoryEntries(directoryReader);
		const filePaths = entries.map((entry: Entry) => {
			return {
				storageUnit: { ...this.storageUnit },
				filePath: trimSlashesAndDots(`${directoryPath.filePath}/${entry.name}`),
			};
		});
		return filePaths;
	}

	public async getFile(filePath: IFilePath): Promise<IFile | null> {
		const fs = await this.getFileSystem(filePath.storageUnit);
		try {
			const fileEntry = await this.getFileEntry(fs, filePath.filePath);
			const localUri = fileEntry.toURL();
			const response = await this.fetchWithFailoverUsingMetadata(localUri, { method: 'HEAD' });
			let lastModifiedAt = response.headers.has('If-Modified-Since')
				? Math.floor(new Date(response.headers.get('If-Modified-Since')!).valueOf() / 1000)
				: undefined;
			lastModifiedAt = response.headers.has('Last-Modified')
				? Math.floor(new Date(response.headers.get('Last-Modified')!).valueOf() / 1000)
				: lastModifiedAt;
			const sizeBytes = response.headers.has('Content-Length') ? parseInt(response.headers.get('Content-Length')!) : undefined;
			const mimeType = response.headers.has('Content-Type')
				? response.headers.get('Content-Type')!
				: await this.detectMimeTypeOfUri(localUri);
			return {
				localUri,
				createdAt: undefined, // it is not detectable
				lastModifiedAt,
				sizeBytes,
				mimeType,
			};
		} catch (error) {
			return null;
		}
	}

	public async readFile(filePath: IFilePath): Promise<string> {
		return await this.getFileText(filePath);
	}

	public async writeFile(filePath: IFilePath, contents: string) {
		const fileTextBase64Encoded = Buffer.from(contents).toString('base64');
		const playlistEncodedUri = `data:application/json;base64,${fileTextBase64Encoded}`;
		await this.downloadFile(filePath, playlistEncodedUri);
	}

	public async appendFile(filePath: IFilePath, contents: string): Promise<void> {
		const fileContents = (await this.exists(filePath)) ? await this.readFile(filePath) : '';
		const newContents = fileContents + contents;
		await this.writeFile(filePath, newContents);
	}

	public async exists(filePath: IFilePath): Promise<boolean> {
		const fs = await this.getFileSystem(filePath.storageUnit);
		try {
			await this.getEntry(fs, filePath.filePath);
			return true;
		} catch (error) {
			return false;
		}
	}

	public async downloadFile(filePath: IFilePath, sourceUri: string, headers?: IHeaders): Promise<void> {
		const response = await this.fetchOrDecodeDataUri(this.window, sourceUri, { headers });
		if (!response.ok) {
			throw new Error(`Not OK response during downloadFile ${sourceUri}`);
		}
		const blob = await response.blob();
		const fs = await this.getFileSystem(filePath.storageUnit);
		const fileEntry = await this.getFileEntry(fs, filePath.filePath, { create: true });
		await this.writeBlobToFileEntry(fileEntry, blob);

		try {
			const localUri = fileEntry.toURL();
			await this.setMetadata(localUri, {
				createdAt: new Date().valueOf(),
				originalSourceUri: sourceUri,
				originalRequestHeaders: headers,
				originalResponseHeaders: this.getHeadersObject(response.headers),
			});
		} catch (error) {
			debug('Cannot set metadata', filePath, sourceUri, error);
		}
	}

	public async uploadFile(filePath: IFilePath, uri: string, formKey: string, headers?: { [key: string]: string }): Promise<string> {
		const blob = await this.getFileBlob(filePath);
		const formData = new FormData();
		formData.append(formKey, blob);
		const response = await fetch(uri, { method: 'POST', body: formData, headers });
		if (response.ok) {
			try {
				return await response.text();
			} catch (e) {
				// ignore if can't read the body (probably form data)
				return '';
			}
		}
		throw new Error('Error while reading file');
	}

	public async deleteFile(filePath: IFilePath, recursive: boolean): Promise<void> {
		if (recursive && (await this.isDirectory(filePath))) {
			const subFilePaths = await this.listFiles(filePath);
			await Promise.all(subFilePaths.map((subFilePath: IFilePath) => this.deleteFile(subFilePath, true)));
		}
		const fs = await this.getFileSystem(filePath.storageUnit);
		const fileEntry = await this.getEntry(fs, filePath.filePath);
		await new Promise<void>((resolve: () => void, reject: (error: Error) => void) => {
			fileEntry.remove(
				() => resolve(),
				(error: Error) => reject(error),
			);
		});

		try {
			const localUri = fileEntry.toURL();
			await this.deleteMetadata(localUri);
		} catch (error) {
			debug('Cannot delete metadata', filePath, error);
		}
	}

	public async copyFile(sourceFilePath: IFilePath, destinationFilePath: IFilePath, options: ICopyFileOptions = {}): Promise<void> {
		if (await this.exists(destinationFilePath)) {
			if (options.overwrite) {
				await this.deleteFile(destinationFilePath, true);
			} else {
				throw new Error(`Cannot copy file to existing path ${destinationFilePath}`);
			}
		}
		const sourceFs = await this.getFileSystem(sourceFilePath.storageUnit);
		const destinationFs = await this.getFileSystem(destinationFilePath.storageUnit);
		const fileEntry = await this.getEntry(sourceFs, sourceFilePath.filePath);
		const destinationDirectoryEntry = await this.getDirectoryEntry(destinationFs, path.dirname(destinationFilePath.filePath));
		const destinationFileName = path.basename(destinationFilePath.filePath);
		await new Promise<void>((resolve: () => void, reject: (error: Error) => void) => {
			fileEntry.copyTo(
				destinationDirectoryEntry,
				destinationFileName,
				() => resolve(),
				(error: Error) => reject(error),
			);
		});

		try {
			await this.copyOrMoveMetadata(fileEntry, destinationFs, destinationFilePath, 'copyTo');
		} catch (error) {
			debug('Cannot copy metadata', sourceFilePath, destinationFilePath, error);
		}
	}

	public async moveFile(sourceFilePath: IFilePath, destinationFilePath: IFilePath, options: IMoveFileOptions = {}): Promise<void> {
		if (await this.exists(destinationFilePath)) {
			if (options.overwrite) {
				await this.deleteFile(destinationFilePath, true);
			} else {
				throw new Error(`Cannot move file to existing path ${destinationFilePath}`);
			}
		}
		const sourceFs = await this.getFileSystem(sourceFilePath.storageUnit);
		const destinationFs = await this.getFileSystem(destinationFilePath.storageUnit);
		const fileEntry = await this.getEntry(sourceFs, sourceFilePath.filePath);
		const destinationDirectoryEntry = await this.getDirectoryEntry(destinationFs, path.dirname(destinationFilePath.filePath));
		const destinationFileName = path.basename(destinationFilePath.filePath);
		await new Promise<void>((resolve: () => void, reject: (error: Error) => void) => {
			fileEntry.moveTo(
				destinationDirectoryEntry,
				destinationFileName,
				() => resolve(),
				(error: Error) => reject(error),
			);
		});

		try {
			await this.copyOrMoveMetadata(fileEntry, destinationFs, destinationFilePath, 'moveTo');
		} catch (error) {
			debug('Cannot move metadata', sourceFilePath, destinationFilePath, error);
		}
	}

	public link(_sourceFilePath: IFilePath, _destinationFilePath: IFilePath): Promise<void> {
		throw new NotSupportedMethodError('link');
	}

	public async createDirectory(directoryPath: IFilePath): Promise<void> {
		const fs = await this.getFileSystem(directoryPath.storageUnit);
		await this.getDirectoryEntry(fs, directoryPath.filePath, { create: true, exclusive: true });
	}

	public async isDirectory(filePath: IFilePath): Promise<boolean> {
		const fs = await this.getFileSystem(filePath.storageUnit);
		const entry = await this.getEntry(fs, filePath.filePath);
		return entry.isDirectory;
	}

	public async extractFile(archiveFilePath: IFilePath, destinationDirectoryPath: IFilePath, method: string): Promise<void> {
		if (method !== 'zip') {
			throw new Error(`Unsupported archive method ${method}`);
		}
		const archiveFile = await this.getFile(archiveFilePath);
		if (!archiveFile) {
			throw new Error(`Archive file ${archiveFilePath.filePath} in storage ${archiveFilePath.storageUnit} doesn't exist.`);
		}
		if (!(await this.exists(destinationDirectoryPath))) {
			await this.createDirectory(destinationDirectoryPath);
		}
		const destinationFs = await this.getFileSystem(destinationDirectoryPath.storageUnit);
		await extractZipFile(
			archiveFile.localUri,
			(relativeFilePath: string) => this.ensureDirectory(destinationFs, destinationDirectoryPath.filePath + '/' + relativeFilePath),
			async (relativeFilePath: string, blob: Blob) => {
				const filePath = destinationDirectoryPath.filePath + '/' + relativeFilePath;
				const fileEntry = await this.getFileEntry(destinationFs, filePath, { create: true });
				await this.writeBlobToFileEntry(fileEntry, blob);
			},
		);
	}

	public async createArchive(_archiveFilePath: IFilePath, _archiveEntries: IFilePath[]): Promise<void> {
		throw new NotSupportedMethodError('createArchive');
	}

	public async getFileChecksum(filePath: IFilePath, hashType: HashAlgorithm): Promise<string> {
		const file = await this.getFile(filePath);
		if (!file) {
			throw new Error(`File ${filePath.filePath} in storage ${filePath.storageUnit} doesn't exist.`);
		}
		return await getChecksumOfFile(this.window, file.localUri, hashType);
	}

	public async listStorageUnits(): Promise<IStorageUnit[]> {
		const storageUnit = this.storageUnit;
		const fs = await this.getFileSystem(storageUnit);
		const usedSpace = await this.getDirectoryUsageSize(fs.root as DirectoryEntry);
		storageUnit.freeSpace = storageUnit.usableSpace = storageUnit.capacity - usedSpace;
		return [storageUnit];
	}

	public onStorageUnitsChanged(_listener: () => void) {
		// Storages never change. Only one available
	}

	public async getArchiveInfo(archiveFilePath: IFilePath): Promise<IArchiveInfo> {
		const file = await this.getFile(archiveFilePath);
		if (!file) {
			throw new Error(`File ${archiveFilePath.filePath} doesn't exist.`);
		}
		const fileData = await this.fetchWithFailover(file.localUri, { method: 'GET' });
		const uncompressedSize = await getZipExtractSize(await fileData.blob());
		return { uncompressedSize };
	}

	public async wipeout(): Promise<void> {
		await this.deleteFile({ storageUnit: this.storageUnit, filePath: 'data' }, true);
	}

	private async ensureDirectory(fs: FileSystem, directoryPath: string) {
		const parentDirectoryPath = path.dirname(directoryPath);
		if (parentDirectoryPath !== '.') {
			await this.ensureDirectory(fs, parentDirectoryPath);
		}
		await this.getDirectoryEntry(fs, directoryPath, { create: true }); // ensure dir
	}

	private async getDirectoryUsageSize(directoryEntry: DirectoryEntry) {
		const directoryReader = directoryEntry.createReader();
		const entries = await this.readDirectoryEntries(directoryReader);
		const totalUsage: number = await entries.reduce(async (usage: Promise<number>, entry: Entry) => {
			if (entry.isDirectory) {
				return (await usage) + (await this.getDirectoryUsageSize(entry as DirectoryEntry));
			} else {
				const fileMetadata = await this.getFileMatadata(entry as FileEntry);
				return (await usage) + fileMetadata.size;
			}
		}, Promise.resolve(0));
		return totalUsage;
	}

	private async getFileText(filePath: IFilePath): Promise<string> {
		const response = await this.fetchFile(filePath);
		if (response.ok && response.body !== null) {
			return await response.text();
		}
		throw new Error('Error while reading file');
	}

	private async getFileBlob(filePath: IFilePath): Promise<Blob> {
		const response = await this.fetchFile(filePath);
		if (response.ok && response.body !== null) {
			return await response.blob();
		}
		throw new Error('Error while reading file');
	}

	private async fetchFile(filePath: IFilePath) {
		const fs = await this.getFileSystem(filePath.storageUnit);
		try {
			const fileEntry = await this.getFileEntry(fs, filePath.filePath);
			const localUri = fileEntry.toURL();
			return await this.fetchWithFailoverUsingMetadata(localUri);
		} catch (error) {
			throw new Error(`File not found: ${error.message}`);
		}
	}

	private async getFileSystem(storageUnit: IStorageUnit) {
		if (storageUnit.type !== this.storageUnit.type) {
			throw new Error(`Unknown storage unit type ${storageUnit.type}`);
		}
		if (this.fileSystem) {
			return this.fileSystem;
		}
		const requestFileSystem = this.window.requestFileSystem || this.window.webkitRequestFileSystem;
		this.fileSystem = await new Promise((resolve: (fileSystem: FileSystem) => void, reject: (error: Error) => void) => {
			requestFileSystem(
				this.fileSystemType,
				this.storageUnit.capacity,
				(fileSystem: FileSystem) => resolve(fileSystem),
				(error: Error) => reject(error),
			);
		});
		return this.fileSystem;
	}

	private async getEntry(fs: FileSystem, filePath: string, flags?: Flags) {
		try {
			return await this.getFileEntry(fs, filePath, flags);
		} catch (error) {
			return await this.getDirectoryEntry(fs, filePath, flags);
		}
	}

	private getFileEntry(fs: FileSystem, filePath: string, flags?: Flags) {
		return new Promise((resolve: (entry: FileEntry) => void, reject: (error: Error) => void) => {
			fs.root.getFile(
				filePath,
				flags,
				(entry: FileEntry) => resolve(entry),
				(error: Error) => reject(error),
			);
		});
	}

	private getDirectoryEntry(fs: FileSystem, filePath: string, flags?: Flags) {
		return new Promise((resolve: (entry: DirectoryEntry) => void, reject: (error: Error) => void) => {
			fs.root.getDirectory(
				filePath,
				flags,
				(entry: DirectoryEntry) => resolve(entry),
				(error: Error) => reject(error),
			);
		});
	}

	private readDirectoryEntries(directoryReader: DirectoryReader) {
		return new Promise((resolve: (entries: Entry[]) => void, reject: (error: Error) => void) => {
			directoryReader.readEntries(
				(entries: Entry[]) => resolve(entries),
				(error: Error) => reject(error),
			);
		});
	}

	private createWriter(fileEntry: FileEntry) {
		return new Promise((resolve: (fileWriter: FileWriter) => void, reject: (error: Error) => void) => {
			fileEntry.createWriter(
				(fileWriter: FileWriter) => resolve(fileWriter),
				(error: Error) => reject(error),
			);
		});
	}

	private getFileMatadata(entry: FileEntry) {
		return new Promise((resolve: (metadata: Metadata) => void, reject: (error: Error) => void) => {
			entry.getMetadata(
				(metadata: Metadata) => resolve(metadata),
				(error: Error) => reject(error),
			);
		});
	}

	private async writeBlobToFileEntry(fileEntry: FileEntry, blob: Blob) {
		const fileWriter = await this.createWriter(fileEntry);
		const truncateEndPromise = new Promise<void>((resolve: () => void, reject: (error: Error) => void) => {
			fileWriter.onwriteend = (_event: ProgressEvent) => resolve();
			fileWriter.onerror = (_event: ProgressEvent) => reject(new Error(`Error during truncate file ${fileEntry.fullPath} from blob`));
		});
		fileWriter.truncate(0);
		await truncateEndPromise;
		const writeEndPromise = new Promise<void>((resolve: () => void, reject: (error: Error) => void) => {
			fileWriter.onwriteend = (_event: ProgressEvent) => resolve();
			fileWriter.onerror = (_event: ProgressEvent) => reject(new Error(`Error during write file ${fileEntry.fullPath} from blob`));
		});
		fileWriter.write(blob);
		await writeEndPromise;
	}

	private async detectMimeTypeOfUri(localUri: string) {
		const response = await this.fetchWithFailoverUsingMetadata(localUri);
		const blob = await response.blob();
		return blob.type;
	}

	private async fetchWithFailoverUsingMetadata(localUri: string, options: { method?: 'GET' | 'HEAD' } = {}) {
		const response = await this.fetchWithFailover(localUri, options);
		const metadata = await this.getMetadata(localUri);
		const mixedHeaders: IHeaders = {
			...metadata?.originalResponseHeaders,
			...(metadata ? { 'last-modified': new Date(metadata.createdAt).toString() } : {}),
			...(response.headers ? this.getHeadersObject(response.headers) : {}),
		};
		const overwrittenHeaders = this.getSimpleHeaders(mixedHeaders);
		const extendedResponse = Object.assign(Object.create(Object.getPrototypeOf(response)), response, { headers: overwrittenHeaders });
		return extendedResponse;
	}

	private async fetchWithFailover(localUri: string, options: { method?: 'GET' | 'HEAD' } = {}) {
		try {
			return await this.fetchOrDecodeDataUri(this.window, localUri, options);
		} catch (error) {
			debug('Original fetch failed', localUri, options, error);
			return await this.simpleFetch(localUri, options);
		}
	}

	private simpleFetch(localUri: string, options: { method?: 'GET' | 'HEAD' } = {}) {
		return new Promise((resolve: (response: ISimpleResponse) => void, reject: (error: Error) => void) => {
			// defaults
			options.method = options.method ?? 'GET';

			const xhr = new XMLHttpRequest();
			xhr.open(options.method, localUri);
			if (options.method === 'GET') {
				xhr.responseType = 'arraybuffer';
			}
			xhr.onerror = (event: ProgressEvent) => {
				reject(new Error(`Network request failed for ${localUri}. Loaded ${event.loaded} of ${event.total} bytes.`));
			};
			xhr.onload = async () => {
				const statusGroup = Math.floor(xhr.status / 100);
				const headers = _.fromPairs(
					xhr
						.getAllResponseHeaders()
						.split('\n')
						.map((header: string) => {
							const pair = header.split(/: */);
							return [pair[0].toLowerCase(), pair[1]];
						}),
				);
				resolve({
					async blob() {
						const blob = new Blob([new Uint8Array(xhr.response, 0, xhr.response.byteLength)]);
						return blob;
					},
					async text() {
						const responseUint16Array = new Uint16Array(xhr.response);
						const responseNumberArray = Array.from(responseUint16Array);
						return String.fromCharCode.apply(null, responseNumberArray);
					},
					ok: statusGroup === 2 || statusGroup === 3,
					body: xhr.response,
					headers: this.getSimpleHeaders(headers),
				});
			};

			xhr.send();
		});
	}

	private async getMetadata(localUri: string): Promise<IMetadata | null> {
		const fs = await this.getFileSystem(this.storageUnit);
		const metadataFilePath = this.getMetadataFilePath(localUri);
		try {
			const fileEntry = await this.getFileEntry(fs, metadataFilePath);
			const metadataLocalUri = fileEntry.toURL();
			const metadataResponse = await this.fetchOrDecodeDataUri(window, metadataLocalUri);
			if (metadataResponse.ok && metadataResponse.body !== null) {
				return await metadataResponse.json();
			}
		} catch (error) {
			debug(`Failed to read metadata`, localUri, error);
			return null;
		}
		return null;
	}

	private async setMetadata(localUri: string, metadata: IMetadata) {
		const fileTextBase64Encoded = Buffer.from(JSON.stringify(metadata)).toString('base64');
		const playlistEncodedUri = `data:application/json;base64,${fileTextBase64Encoded}`;
		const response = await this.fetchOrDecodeDataUri(window, playlistEncodedUri);
		const blob = await response.blob();
		const fs = await this.getFileSystem(this.storageUnit);
		await this.getDirectoryEntry(fs, METADATA_DIRECTORY, { create: true }); // ensure metadata dir
		const metadataFilePath = this.getMetadataFilePath(localUri);
		const fileEntry = await this.getFileEntry(fs, metadataFilePath, { create: true });
		await this.writeBlobToFileEntry(fileEntry, blob);
	}

	private async copyOrMoveMetadata(
		originalFileEntry: DirectoryEntry | FileEntry,
		destinationFs: FileSystem,
		destinationFilePath: IFilePath,
		method: 'copyTo' | 'moveTo',
	) {
		const fs = await this.getFileSystem(this.storageUnit);
		const metadataSourceFileEntry = await this.getEntry(fs, this.getMetadataFilePath(originalFileEntry.toURL()));
		const metadataDirectoryEntry = await this.getDirectoryEntry(fs, METADATA_DIRECTORY);
		const destinationFileEntry = await this.getFileEntry(destinationFs, destinationFilePath.filePath);
		const metadataNewFileName = path.basename(this.getMetadataFilePath(destinationFileEntry.toURL()));
		await new Promise<void>((resolve: () => void, reject: (error: Error) => void) => {
			metadataSourceFileEntry[method](
				metadataDirectoryEntry,
				metadataNewFileName,
				() => resolve(),
				(error: Error) => reject(error),
			);
		});
	}

	private async deleteMetadata(localUri: string) {
		const fs = await this.getFileSystem(this.storageUnit);
		const fileEntry = await this.getEntry(fs, this.getMetadataFilePath(localUri));
		await new Promise<void>((resolve: () => void, reject: (error: Error) => void) => {
			fileEntry.remove(
				() => resolve(),
				(error: Error) => reject(error),
			);
		});
	}

	private getMetadataFilePath(localUri: string) {
		const metadataFilePath = `${METADATA_DIRECTORY}/${checksumString(localUri)}`;
		return metadataFilePath;
	}

	private getSimpleHeaders(headersObject: IHeaders): ISimpleHeaders {
		return {
			has(name: string) {
				return typeof headersObject[name.toLowerCase()] !== 'undefined';
			},
			get(name: string) {
				return headersObject[name.toLowerCase()];
			},
			forEach(callbackfn: (value: string, key: string) => void) {
				for (const key in headersObject) {
					callbackfn(headersObject[key], key);
				}
			},
		};
	}

	private getHeadersObject(headers: ISimpleHeaders | Headers) {
		const headersObject: IHeaders = {};
		headers.forEach((value: string, key: string) => (headersObject[key.toLowerCase()] = value));
		return headersObject;
	}

	private async requestFileSystem(fileSystemType: number, storageSizeBytes: number) {
		let storageInfo: {
			requestQuota(requestedBytes: number, successCallback: (grantedBytes: number) => void, errorCallback: (error: Error) => void): void;
			queryUsageAndQuota(successCallback: (usedBytes: number, availableBytes: number) => void, errorCallback: (error: Error) => void): void;
		};
		if (fileSystemType === this.window.PERSISTENT) {
			storageInfo = (this.window.navigator as any).webkitPersistentStorage;
		} else if (fileSystemType === this.window.TEMPORARY) {
			storageInfo = (this.window.navigator as any).webkitTemporaryStorage;
		} else {
			console.warn(`Unknown FS type ${fileSystemType}`);
			return 0;
		}

		let grantedSizeBytes: number = await new Promise((resolve: (grantedSizeBytes: number) => void, reject: (error: Error) => void) => {
			try {
				storageInfo.queryUsageAndQuota(
					(usedBytes: number, availableBytes: number) => resolve(usedBytes + availableBytes),
					(error: Error) => reject(error),
				);
			} catch (error) {
				reject(error);
			}
		});

		if (grantedSizeBytes === 0) {
			grantedSizeBytes = await new Promise((resolve: (grantedSizeBytes: number) => void, reject: (error: Error) => void) => {
				try {
					storageInfo.requestQuota(
						storageSizeBytes,
						(grantedBytes: number) => resolve(grantedBytes),
						(error: Error) => reject(error),
					);
				} catch (error) {
					reject(error);
				}
			});
		}

		if (storageSizeBytes !== grantedSizeBytes) {
			debug(`Requested for ${storageSizeBytes} bytes of FS but got only ${grantedSizeBytes} bytes`);
		}

		this.storageUnit = {
			type: fileSystemType.toString(),
			capacity: grantedSizeBytes,
			freeSpace: 0,
			usableSpace: 0,
			removable: false,
		};

		await this.getFileSystem(this.storageUnit); // Just check FS works
	}

	// XMLHttpRequest (inside of fetch polyfill) can't handle datauris
	private async fetchOrDecodeDataUri(window: Window, ...[req, init]: Parameters<typeof fetch>) {
		if (typeof req !== 'string' || !req.startsWith('data:')) {
			return await window.fetch(req, init);
		} else {
			const url = new URL(req);
			let [, data] = url.pathname.split(';');
			if (data.startsWith('base64,')) {
				data = atob(data.slice('base64,'.length));
			}
			const blob = new Blob([data]);
			return new Response(blob);
		}
	}
}
