import { IFile, IFilePath, IHeaders, IStorageUnit } from '../NativeDevice/fileSystem';
import HashAlgorithm from '../NativeDevice/HashAlgorithm';
import IFileSystem, { IArchiveInfo, ICopyFileOptions, IMoveFileOptions } from '../NativeDevice/IFileSystem';
import { calculateReservedSpace, validateFreeSpace } from './fileSystemHelper';
import { createWindowHttpHeadFetcher, IHttpHeadFetcher } from './httpHeadFetchers';

export class FileSystemWithReservedSpace implements IFileSystem {
	constructor(
		private readonly fileSystem: IFileSystem,
		private readonly httpHeadFetcher: IHttpHeadFetcher,
		private readonly reservedPercentage: number,
	) {}

	public async writeFile(filePath: IFilePath, contents: string): Promise<void> {
		const requestedFileSystem = await this.getStorageUnitByType(filePath.storageUnit.type);
		if (!(await validateFreeSpace(requestedFileSystem, filePath.filePath, Buffer.byteLength(contents, 'utf-8')))) {
			throw new Error('Not enough free space.');
		}
		return await this.fileSystem.writeFile(filePath, contents);
	}

	public async listStorageUnits(): Promise<IStorageUnit[]> {
		const storageUnits = await this.fileSystem.listStorageUnits();
		return storageUnits.map((storageUnit: IStorageUnit) => {
			const reservedSpace = calculateReservedSpace(storageUnit.capacity, this.reservedPercentage);
			// Don't go below zero, in case we're already limited by platform.
			storageUnit.usableSpace = Math.max(0, storageUnit.usableSpace - reservedSpace);
			return storageUnit;
		});
	}

	public async onStorageUnitsChanged(listener: () => void): Promise<void> {
		return this.fileSystem.onStorageUnitsChanged(listener);
	}

	public async listFiles(directoryPath: IFilePath): Promise<IFilePath[]> {
		return await this.fileSystem.listFiles(directoryPath);
	}

	public async getFile(filePath: IFilePath): Promise<IFile | null> {
		return await this.fileSystem.getFile(filePath);
	}

	public async readFile(filePath: IFilePath): Promise<string> {
		return await this.fileSystem.readFile(filePath);
	}

	public async appendFile(filePath: IFilePath, contents: string): Promise<void> {
		const requestedFileSystem = await this.getStorageUnitByType(filePath.storageUnit.type);
		if (!(await validateFreeSpace(requestedFileSystem, filePath.filePath, Buffer.byteLength(contents, 'utf-8')))) {
			throw new Error('Not enough free space.');
		}
		return await this.fileSystem.appendFile(filePath, contents);
	}

	public async exists(filePath: IFilePath): Promise<boolean> {
		return await this.fileSystem.exists(filePath);
	}

	public async downloadFile(filePath: IFilePath, sourceUri: string, headers?: IHeaders | undefined): Promise<void> {
		let contentLength: string | number | undefined;
		try {
			const response = await this.httpHeadFetcher(sourceUri, headers);
			contentLength = response.headers['content-length'];
		} catch (error) {
			console.warn(`Failed to process HEAD request on downloaded file ${sourceUri}`);
		}
		if (contentLength === undefined) {
			// This can't be determined due to functionality of gzip/chunk-enconding. https://www.w3.org/Protocols/rfc2616/rfc2616-sec4.html#sec4.4
			console.warn(
				`No Content-Length header during downloadFile ${sourceUri}. In that case, we can not determine the full functionality of checking free space.`,
			);
			contentLength = 0;
		}
		const requestedFileSystem = await this.getStorageUnitByType(filePath.storageUnit.type);
		if (!(await validateFreeSpace(requestedFileSystem, filePath.filePath, Number(contentLength)))) {
			throw new Error('Not enough free space.');
		}
		return await this.fileSystem.downloadFile(filePath, sourceUri, headers);
	}

	public async uploadFile(filePath: IFilePath, uri: string, formKey: string, headers?: IHeaders | undefined): Promise<string | undefined> {
		return await this.fileSystem.uploadFile(filePath, uri, formKey, headers);
	}

	public async deleteFile(filePath: IFilePath, recursive: boolean): Promise<void> {
		return await this.fileSystem.deleteFile(filePath, recursive);
	}

	public async copyFile(sourceFilePath: IFilePath, destinationFilePath: IFilePath, options?: ICopyFileOptions | undefined): Promise<void> {
		const file = await this.fileSystem.getFile(sourceFilePath);
		if (file !== null) {
			const requestedFileSystem = await this.getStorageUnitByType(destinationFilePath.storageUnit.type);
			if (!(await validateFreeSpace(requestedFileSystem, destinationFilePath.filePath, file?.sizeBytes))) {
				throw new Error('Not enough free space.');
			}
		}
		return await this.fileSystem.copyFile(sourceFilePath, destinationFilePath, options);
	}

	public async moveFile(sourceFilePath: IFilePath, destinationFilePath: IFilePath, options?: IMoveFileOptions | undefined): Promise<void> {
		const file = await this.fileSystem.getFile(sourceFilePath);
		if (file !== null && sourceFilePath.storageUnit.type !== destinationFilePath.storageUnit.type) {
			const requestedFileSystem = await this.getStorageUnitByType(destinationFilePath.storageUnit.type);
			if (!(await validateFreeSpace(requestedFileSystem, destinationFilePath.filePath, file.sizeBytes))) {
				throw new Error('Not enough free space.');
			}
		}
		return await this.fileSystem.moveFile(sourceFilePath, destinationFilePath, options);
	}

	public async link(sourceFilePath: IFilePath, destinationFilePath: IFilePath): Promise<void> {
		return await this.fileSystem.link(sourceFilePath, destinationFilePath);
	}

	public async getFileChecksum(filePath: IFilePath, hashType: HashAlgorithm): Promise<string> {
		return await this.fileSystem.getFileChecksum(filePath, hashType);
	}

	public async extractFile(archiveFilePath: IFilePath, destinationDirectoryPath: IFilePath, method: string): Promise<void> {
		const { uncompressedSize } = await this.fileSystem.getArchiveInfo(archiveFilePath);
		const requestedFileSystem = await this.getStorageUnitByType(destinationDirectoryPath.storageUnit.type);
		if (!(await validateFreeSpace(requestedFileSystem, destinationDirectoryPath.filePath, uncompressedSize))) {
			throw new Error('Not enough free space.');
		}
		return await this.fileSystem.extractFile(archiveFilePath, destinationDirectoryPath, method);
	}

	public async createArchive(archiveFilePath: IFilePath, archiveEntries: IFilePath[]): Promise<void> {
		const requestedFileSystem = await this.getStorageUnitByType(archiveFilePath.storageUnit.type);
		const expectedArchiveSize = await new Promise(async (resolve: (finalArchiveSize: number) => void, reject: (error: Error) => void) => {
			try {
				let finalArchiveSize = 0;
				for (const entry of archiveEntries) {
					const file = await this.fileSystem.getFile(entry);
					if (file !== null) {
						finalArchiveSize += file.sizeBytes ?? 0;
					}
				}
				resolve(finalArchiveSize);
			} catch (error) {
				reject(error);
			}
		});
		if (!(await validateFreeSpace(requestedFileSystem, archiveFilePath.filePath, expectedArchiveSize))) {
			throw new Error('Not enough free space.');
		}
		return await this.fileSystem.createArchive(archiveFilePath, archiveEntries);
	}

	public async createDirectory(directoryPath: IFilePath): Promise<void> {
		return await this.fileSystem.createDirectory(directoryPath);
	}

	public async isDirectory(filePath: IFilePath): Promise<boolean> {
		return await this.fileSystem.isDirectory(filePath);
	}

	public async getArchiveInfo(archiveFilePath: IFilePath): Promise<IArchiveInfo> {
		return await this.fileSystem.getArchiveInfo(archiveFilePath);
	}

	public async wipeout(): Promise<void> {
		return await this.fileSystem.wipeout();
	}

	/** @deprecated Test only. */
	public getExtendedFileSystem(): IFileSystem {
		return this.fileSystem;
	}

	private async getStorageUnitByType(type: string): Promise<IStorageUnit> {
		const storageUnits = await this.listStorageUnits();
		const foundStorageUnit = storageUnits.find((storageUnit) => storageUnit.type === type);
		if (foundStorageUnit === undefined) {
			throw new Error(`StorageUnit for type ${type} not found`);
		}
		return foundStorageUnit;
	}
}

export function createFileSystemWithReservedSpaceWithWindowHttpHeadFetcher(
	fileSystem: IFileSystem,
	window: Pick<Window, 'fetch'>,
	reservedPercentage: number,
): IFileSystem {
	const httpHeadFetcher = createWindowHttpHeadFetcher(window);
	return new FileSystemWithReservedSpace(fileSystem, httpHeadFetcher, reservedPercentage);
}
