import { posix as path } from 'path';
import { cachedOperation, cacheInvalidating, clearCache } from '../Cache/methodCache';
import IFileSystem, { IArchiveInfo, ICopyFileOptions, IMoveFileOptions } from '../NativeDevice/IFileSystem';
import { IFilePath, IFile, IHeaders, IStorageUnit } from '../NativeDevice/fileSystem';
import HashAlgorithm from '../NativeDevice/HashAlgorithm';

const FILE_SYSTEM_CACHE_KEY = 'CachedFileSystem';

/**
 * Proxy file system implementation that caches results in memory for better performance
 */
export default class CachedFileSystem implements IFileSystem {
	constructor(
		private fileSystem: IFileSystem,
		private identification: string,
	) {
		this.fileSystem.onStorageUnitsChanged(() => {
			clearCache(`${FILE_SYSTEM_CACHE_KEY}.${this.identification}.`);
		});
	}

	public async initialize() {
		// do nothing
	}

	public getCacheKey(filePath: IFilePath | null, method: keyof CachedFileSystem) {
		if (filePath === null) {
			return `${FILE_SYSTEM_CACHE_KEY}.${this.identification}.${method}`;
		} else {
			return `${FILE_SYSTEM_CACHE_KEY}.${this.identification}.${method}.${filePath.storageUnit.type}/${filePath.filePath}`;
		}
	}

	public getParentDirectory(filePath: IFilePath) {
		const parentFilePath = path.dirname(filePath.filePath);
		return {
			storageUnit: filePath.storageUnit,
			filePath: parentFilePath === '.' ? '' : parentFilePath,
		};
	}

	@cacheInvalidating((self: CachedFileSystem, filePath: IFilePath) => [
		self.getCacheKey(self.getParentDirectory(filePath), 'listFiles'),
		self.getCacheKey(filePath, 'readFile'),
		self.getCacheKey(filePath, 'getFile'),
		self.getCacheKey(filePath, 'exists'),
		self.getCacheKey(filePath, 'getFileChecksum'),
		self.getCacheKey(filePath, 'getArchiveInfo'),
		self.getCacheKey(null, 'listStorageUnits'),
	])
	public async writeFile(filePath: IFilePath, contents: string): Promise<void> {
		return await this.fileSystem.writeFile(filePath, contents);
	}

	@cacheInvalidating((self: CachedFileSystem, filePath: IFilePath) => [
		self.getCacheKey(self.getParentDirectory(filePath), 'listFiles'),
		self.getCacheKey(filePath, 'readFile'),
		self.getCacheKey(filePath, 'getFile'),
		self.getCacheKey(filePath, 'exists'),
		self.getCacheKey(filePath, 'getFileChecksum'),
		self.getCacheKey(filePath, 'getArchiveInfo'),
		self.getCacheKey(null, 'listStorageUnits'),
	])
	public async appendFile(filePath: IFilePath, contents: string): Promise<void> {
		return await this.fileSystem.appendFile(filePath, contents);
	}

	@cachedOperation((self: CachedFileSystem, filePath: IFilePath) => self.getCacheKey(filePath, 'readFile'))
	public async readFile(filePath: IFilePath): Promise<string> {
		return await this.fileSystem.readFile(filePath);
	}

	@cacheInvalidating(
		(self: CachedFileSystem, _sourceFilePath: IFilePath, destinationFilePath: IFilePath) => [
			self.getCacheKey(self.getParentDirectory(destinationFilePath), 'listFiles'),
			self.getCacheKey(destinationFilePath, 'readFile'),
			self.getCacheKey(destinationFilePath, 'getFile'),
			self.getCacheKey(destinationFilePath, 'exists'),
			self.getCacheKey(destinationFilePath, 'getFileChecksum'),
			self.getCacheKey(destinationFilePath, 'isDirectory'),
			self.getCacheKey(destinationFilePath, 'getArchiveInfo'),
			self.getCacheKey(null, 'listStorageUnits'),
		],
		true, // because it can copy directory
	)
	public async copyFile(sourceFilePath: IFilePath, destinationFilePath: IFilePath, options?: ICopyFileOptions): Promise<void> {
		return await this.fileSystem.copyFile(sourceFilePath, destinationFilePath, options);
	}

	@cachedOperation((self: CachedFileSystem, directoryPath: IFilePath) => self.getCacheKey(directoryPath, 'listFiles'))
	public async listFiles(directoryPath: IFilePath): Promise<IFilePath[]> {
		return await this.fileSystem.listFiles(directoryPath);
	}

	@cachedOperation((self: CachedFileSystem, filePath: IFilePath) => self.getCacheKey(filePath, 'getFile'))
	public async getFile(filePath: IFilePath): Promise<IFile | null> {
		return await this.fileSystem.getFile(filePath);
	}

	@cachedOperation((self: CachedFileSystem, filePath: IFilePath) => self.getCacheKey(filePath, 'exists'))
	public async exists(filePath: IFilePath): Promise<boolean> {
		return await this.fileSystem.exists(filePath);
	}

	@cacheInvalidating((self: CachedFileSystem, destinationFilePath: IFilePath) => [
		self.getCacheKey(self.getParentDirectory(destinationFilePath), 'listFiles'),
		self.getCacheKey(destinationFilePath, 'readFile'),
		self.getCacheKey(destinationFilePath, 'getFile'),
		self.getCacheKey(destinationFilePath, 'exists'),
		self.getCacheKey(destinationFilePath, 'getFileChecksum'),
		self.getCacheKey(destinationFilePath, 'getArchiveInfo'),
		self.getCacheKey(null, 'listStorageUnits'),
	])
	public async downloadFile(destinationFilePath: IFilePath, sourceUri: string, headers?: IHeaders): Promise<void> {
		return await this.fileSystem.downloadFile(destinationFilePath, sourceUri, headers);
	}

	public async uploadFile(filePath: IFilePath, uri: string, formKey: string, headers?: { [key: string]: string }) {
		return await this.fileSystem.uploadFile(filePath, uri, formKey, headers);
	}

	@cacheInvalidating(
		(self: CachedFileSystem, filePath: IFilePath) => [
			self.getCacheKey(self.getParentDirectory(filePath), 'listFiles'),
			self.getCacheKey(filePath, 'readFile'),
			self.getCacheKey(filePath, 'getFile'),
			self.getCacheKey(filePath, 'exists'),
			self.getCacheKey(filePath, 'getFileChecksum'),
			self.getCacheKey(filePath, 'isDirectory'),
			self.getCacheKey(filePath, 'getArchiveInfo'),
			self.getCacheKey(null, 'listStorageUnits'),
		],
		true, // because directory can be deleted
	)
	public async deleteFile(filePath: IFilePath, recursive: boolean): Promise<void> {
		return await this.fileSystem.deleteFile(filePath, recursive);
	}

	@cacheInvalidating(
		(self: CachedFileSystem, sourceFilePath: IFilePath, destinationFilePath: IFilePath) => [
			self.getCacheKey(self.getParentDirectory(destinationFilePath), 'listFiles'),
			self.getCacheKey(destinationFilePath, 'readFile'),
			self.getCacheKey(destinationFilePath, 'getFile'),
			self.getCacheKey(destinationFilePath, 'exists'),
			self.getCacheKey(destinationFilePath, 'getFileChecksum'),
			self.getCacheKey(destinationFilePath, 'isDirectory'),
			self.getCacheKey(destinationFilePath, 'getArchiveInfo'),
			self.getCacheKey(self.getParentDirectory(sourceFilePath), 'listFiles'),
			self.getCacheKey(sourceFilePath, 'readFile'),
			self.getCacheKey(sourceFilePath, 'getFile'),
			self.getCacheKey(sourceFilePath, 'exists'),
			self.getCacheKey(sourceFilePath, 'getFileChecksum'),
			self.getCacheKey(sourceFilePath, 'isDirectory'),
			self.getCacheKey(sourceFilePath, 'getArchiveInfo'),
			self.getCacheKey(null, 'listStorageUnits'),
		],
		true, // because it can move directory
	)
	public async moveFile(sourceFilePath: IFilePath, destinationFilePath: IFilePath, options?: IMoveFileOptions): Promise<void> {
		return await this.fileSystem.moveFile(sourceFilePath, destinationFilePath, options);
	}

	@cacheInvalidating(
		(self: CachedFileSystem, _sourceFilePath: IFilePath, destinationFilePath: IFilePath) => [
			self.getCacheKey(self.getParentDirectory(destinationFilePath), 'listFiles'),
			self.getCacheKey(destinationFilePath, 'readFile'),
			self.getCacheKey(destinationFilePath, 'getFile'),
			self.getCacheKey(destinationFilePath, 'exists'),
			self.getCacheKey(destinationFilePath, 'getFileChecksum'),
			self.getCacheKey(destinationFilePath, 'getArchiveInfo'),
			self.getCacheKey(destinationFilePath, 'isDirectory'),
			self.getCacheKey(null, 'listStorageUnits'),
		],
		true, // because it can link directory
	)
	public async link(sourceFilePath: IFilePath, destinationFilePath: IFilePath): Promise<void> {
		return await this.fileSystem.link(sourceFilePath, destinationFilePath);
	}

	@cachedOperation((self: CachedFileSystem, filePath: IFilePath) => self.getCacheKey(filePath, 'getFileChecksum'))
	public async getFileChecksum(filePath: IFilePath, hashType: HashAlgorithm): Promise<string> {
		return await this.fileSystem.getFileChecksum(filePath, hashType);
	}

	@cacheInvalidating(
		(self: CachedFileSystem, _archiveFilePath: IFilePath, destinationDirectoryPath: IFilePath) => [
			self.getCacheKey(self.getParentDirectory(destinationDirectoryPath), 'listFiles'),
			self.getCacheKey(destinationDirectoryPath, 'readFile'),
			self.getCacheKey(destinationDirectoryPath, 'getFile'),
			self.getCacheKey(destinationDirectoryPath, 'exists'),
			self.getCacheKey(destinationDirectoryPath, 'getFileChecksum'),
			self.getCacheKey(destinationDirectoryPath, 'getArchiveInfo'),
			self.getCacheKey(destinationDirectoryPath, 'isDirectory'),
			self.getCacheKey(null, 'listStorageUnits'),
		],
		true, // because it can extract directory
	)
	public async extractFile(archiveFilePath: IFilePath, destinationDirectoryPath: IFilePath, method: string): Promise<void> {
		return await this.fileSystem.extractFile(archiveFilePath, destinationDirectoryPath, method);
	}

	@cacheInvalidating((self: CachedFileSystem, archiveFilePath: IFilePath) => [
		self.getCacheKey(self.getParentDirectory(archiveFilePath), 'listFiles'),
		self.getCacheKey(archiveFilePath, 'readFile'),
		self.getCacheKey(archiveFilePath, 'getFile'),
		self.getCacheKey(archiveFilePath, 'exists'),
		self.getCacheKey(archiveFilePath, 'getFileChecksum'),
		self.getCacheKey(archiveFilePath, 'getArchiveInfo'),
		self.getCacheKey(null, 'listStorageUnits'),
	])
	public async createArchive(archiveFilePath: IFilePath, archiveEntries: IFilePath[]): Promise<void> {
		return await this.fileSystem.createArchive(archiveFilePath, archiveEntries);
	}

	@cacheInvalidating(
		(self: CachedFileSystem, directoryPath: IFilePath) => [
			self.getCacheKey(self.getParentDirectory(directoryPath), 'listFiles'),
			self.getCacheKey(directoryPath, 'readFile'),
			self.getCacheKey(directoryPath, 'getFile'),
			self.getCacheKey(directoryPath, 'exists'),
			self.getCacheKey(directoryPath, 'getFileChecksum'),
			self.getCacheKey(directoryPath, 'getArchiveInfo'),
			self.getCacheKey(directoryPath, 'isDirectory'),
		],
		true,
	)
	public async createDirectory(directoryPath: IFilePath): Promise<void> {
		return await this.fileSystem.createDirectory(directoryPath);
	}

	@cachedOperation((self: CachedFileSystem, filePath: IFilePath) => self.getCacheKey(filePath, 'isDirectory'))
	public async isDirectory(filePath: IFilePath): Promise<boolean> {
		return await this.fileSystem.isDirectory(filePath);
	}

	@cachedOperation((self: CachedFileSystem) => self.getCacheKey(null, 'listStorageUnits'))
	public async listStorageUnits(): Promise<IStorageUnit[]> {
		return await this.fileSystem.listStorageUnits();
	}

	@cachedOperation((self: CachedFileSystem, archiveFilePath: IFilePath) => self.getCacheKey(archiveFilePath, 'getArchiveInfo'))
	public async getArchiveInfo(archiveFilePath: IFilePath): Promise<IArchiveInfo> {
		return await this.fileSystem.getArchiveInfo(archiveFilePath);
	}

	public onStorageUnitsChanged(listener: () => void): void {
		return this.fileSystem.onStorageUnitsChanged(listener);
	}

	@cacheInvalidating(
		(self: CachedFileSystem) => [
			self.getCacheKey(null, 'listFiles'),
			self.getCacheKey(null, 'readFile'),
			self.getCacheKey(null, 'getFile'),
			self.getCacheKey(null, 'exists'),
			self.getCacheKey(null, 'getFileChecksum'),
			self.getCacheKey(null, 'isDirectory'),
			self.getCacheKey(null, 'listStorageUnits'),
			self.getCacheKey(null, 'getArchiveInfo'),
		],
		true,
	)
	public async wipeout(): Promise<void> {
		return this.fileSystem.wipeout();
	}
}
