import { ActiveAppletDispatchCommand, DispatchCommandToApplet } from '@signageos/actions/dist/Applet/activeAppletActions';
import { bindAndTakeEveryOnlyWhenPlatform } from '../Socket/socketActionCreator';
import { SagaIteratorWithPromises } from '../ReduxSaga/extendedEffects';
import { call, put } from 'redux-saga/effects';

function getExperimentalActionType(actionType: string) {
	return `sos.experimental.${actionType}`;
}

/**
 * The feature for DispatchCommandToApplet is hijacked for experimental purposes.
 * This function is same as takeEvery, but it will also listen to DispatchCommandToApplet with experimental prefix.
 * So the server can send the experimental action to the client even if it is not implemented on the server yet.
 *
 * To perform the feature from server, send Applet Command (POST https://api.signageos.io/v1/device/:deviceUid/applet/:appletUid/command)
 * in the following format:
 * ```json
 * {
 *    "type": "sos.experimental.<actionType>",
 *    "command": {
 *       "foo": "<foo param>",
 *       "bar": "<bar param>"
 *    }
 * }
 * ```
 *
 * @deprecated Replace usage with `bindAndTakeEveryOnlyWhenPlatform` when the feature is implemented on the server.
 * @deprecated This function is not really deprecated and will not be removed, but the usage should by minimized.
 */
export function* bindAndTakeEveryOnlyWhenPlatformExperimental<T extends { type: string }>(
	actionType: T['type'],
	callback: (action: T) => void,
) {
	yield bindAndTakeEveryOnlyWhenPlatform(DispatchCommandToApplet, function* (action: DispatchCommandToApplet): SagaIteratorWithPromises {
		if (action.command.type === getExperimentalActionType(actionType)) {
			yield call(callback, {
				...action.command,
				type: actionType,
			});
		}
	});
	// For forward compatibility, also listen to the action without the experimental prefix.
	yield bindAndTakeEveryOnlyWhenPlatform<string, (action: { type: string }) => SagaIteratorWithPromises>(
		actionType,
		function* (action: T): SagaIteratorWithPromises {
			yield call(callback, action);
		},
	);
}

/**
 * The feature for ActiveAppletDispatchCommand is hijacked for experimental purposes.
 * This function is same as put, but it will also send the action with experimental prefix.
 * So the device can send the experimental action to the server even if it is not implemented on the server yet.
 *
 * To perform the feature from device, get Applet Command (GET https://api.signageos.io/v1/device/:deviceUid/applet/:appletUid/command)
 *
 * @deprecated Replace usage with `put` when the feature is implemented on the server.
 * @deprecated This function is not really deprecated and will not be removed, but the usage should by minimized.
 */
export function* putExperimental<T extends { type: string }>(action: T) {
	yield put<ActiveAppletDispatchCommand>({
		type: ActiveAppletDispatchCommand,
		appletUid: '',
		timingChecksum: '',
		command: {
			...action,
			type: getExperimentalActionType(action.type),
		},
	});
	// For forward compatibility, also put the action without the experimental prefix.
	yield put<T>(action);
}
