import AuthenticationFailReason from '@signageos/actions/dist/Authentication/AuthenticationFailReason';
import { AuthenticationFailed, AuthenticationSucceed } from '@signageos/actions/dist/Authentication/authenticationActions';
import { NotifyDeviceApplicationVersion } from '@signageos/actions/dist/Device/Application/deviceApplicationActions';
import { NotifyDeviceFrontDisplayVersion } from '@signageos/actions/dist/Device/FrontDisplay/deviceFrontDisplayActions';
import { RegisterDevice, UpdateDeviceInfo } from '@signageos/actions/dist/Device/deviceActions';
import Debug from 'debug';
import { put } from 'redux-saga/effects';
import IBasicDriver from '../NativeDevice/IBasicDriver';
import { IFrontManagementDriver } from '../NativeDevice/Management/IManagementDriver';
import ManagementCapability from '../NativeDevice/Management/ManagementCapability';
import { bindAndTakeEvery } from '../Socket/socketActionCreator';
import { UpdateSerialNumberSupport } from './deviceActions';

const debug = Debug('@signageos/front-display:Device:deviceSaga');

export function* fallbackDeviceRegistration(
	getNativeDriver: () => IBasicDriver,
	autoVerification?: { organizationUid: string; deviceName?: string } | undefined,
) {
	yield bindAndTakeEvery(AuthenticationFailed, function* (action: AuthenticationFailed) {
		try {
			if (action.reason === AuthenticationFailReason.MISSING_DEVICE) {
				const nativeDriver = getNativeDriver();
				const deviceUid: string = yield nativeDriver.getDeviceUid();
				yield put({
					type: RegisterDevice,
					uid: deviceUid,
					applicationType: nativeDriver.getApplicationType(),
					autoVerification,
				} as RegisterDevice);
			}
		} catch (error) {
			console.error('fallbackDeviceRegistration failed', error);
		}
	});
}
export function* notifyDeviceInfo(getNativeDriver: () => IFrontManagementDriver) {
	try {
		const nativeDriver = getNativeDriver();
		const applicationType = nativeDriver.getApplicationType();
		let deviceModel: string | undefined;
		let serialNumber: string | undefined;
		let brand: string | undefined;
		let osVersion: string | undefined;

		const supportsModel: boolean = yield nativeDriver.managementSupports(ManagementCapability.MODEL);
		if (supportsModel) {
			try {
				deviceModel = yield nativeDriver.systemInfo.getModel();
			} catch (error) {
				console.error(error.message);
			}
		}

		const supportSN: boolean = yield nativeDriver.managementSupports(ManagementCapability.SERIAL_NUMBER);
		yield put<UpdateSerialNumberSupport>({ type: UpdateSerialNumberSupport, supported: supportSN });

		if (supportSN) {
			try {
				serialNumber = yield nativeDriver.systemInfo.getSerialNumber();
			} catch (error) {
				console.error(error.message);
			}
		}

		const supportsBrand: boolean = yield nativeDriver.managementSupports(ManagementCapability.BRAND);
		if (supportsBrand) {
			try {
				brand = yield nativeDriver.systemInfo.getBrand();
			} catch (error) {
				console.error(error.message);
			}
		}

		const supportsOsVersion: boolean = yield nativeDriver.managementSupports(ManagementCapability.OS_VERSION);
		if (supportsOsVersion) {
			try {
				osVersion = yield nativeDriver.systemInfo.getOSVersion();
			} catch (error) {
				console.error(error.message);
			}
		}

		yield put<UpdateDeviceInfo>({
			type: UpdateDeviceInfo,
			applicationType,
			model: deviceModel,
			serialNumber: serialNumber,
			brand,
			osVersion,
		});
	} catch (error) {
		console.error('notifyDeviceInfo failed', error);
	}
}

export function* notifyDeviceProxyInfo(getNativeDriver: () => IFrontManagementDriver) {
	try {
		const nativeDriver = getNativeDriver();
		const supports: boolean = yield nativeDriver.managementSupports(ManagementCapability.PROXY);
		if (supports) {
			const enabled: boolean = yield nativeDriver.proxy.isEnabled();
			const uri: string = yield nativeDriver.proxy.getConnectedTo();

			yield put<UpdateDeviceInfo>({
				type: UpdateDeviceInfo,
				proxy: {
					enabled,
					uri,
				},
			});
		}
	} catch (error) {
		console.error('notifyDeviceProxyInfo failed', error);
	}
}

export function* notifyDeviceFrontDisplayVersion(frontDisplayVersion: string) {
	yield bindAndTakeEvery(AuthenticationSucceed, function* () {
		try {
			yield put({
				type: NotifyDeviceFrontDisplayVersion,
				version: frontDisplayVersion,
			} as NotifyDeviceFrontDisplayVersion);
		} catch (error) {
			console.error('notifyDeviceFrontDisplayVersion failed', error);
		}
	});
}

export function* notifyDeviceApplicationVersionCommon(getNativeDriver: () => IBasicDriver, applicationVersion: string) {
	try {
		const deviceUid: string = yield getNativeDriver().getDeviceUid();
		try {
			yield put({
				type: NotifyDeviceApplicationVersion,
				deviceUid,
				applicationVersion,
			} as NotifyDeviceApplicationVersion);
		} catch (error) {
			debug('No fixed application version', error);
		}
	} catch (error) {
		console.error('notifyDeviceApplicationVersion failed', error);
	}
}
