import { StartApplication, UpdateBaseUrl } from '@signageos/actions/dist/Application/applicationActions';
import { PlatformConnect } from '@signageos/actions/dist/Socket/socketActions';
import wait from '@signageos/lib/dist/Timer/wait';
import Debug from 'debug';
import { put, takeEvery } from 'redux-saga/effects';
import { injectDependencies } from '../../DI/dependencyInjection';
import { DEBUG_CHECK_INTERVAL_MS } from '../../Debug/debugConstants';
import IConfig, { SubscriptionType } from '../../Display/IConfig';
import IBasicDriver from '../../NativeDevice/IBasicDriver';
import { IFrontManagementDriver } from '../../NativeDevice/Management/IManagementDriver';
import Property from '../../Property/Property';
import { IPropertyStorage } from '../../Property/propertyStorage';
import {
	ChangeSubscriptionToOpen,
	ChangeSubscriptionToPlatform,
	ConfigurationChanged,
	IOptionalConfig,
} from './deviceConfigurationActions';
import { checkDeviceConfigurationSaga } from './deviceConfigurationCheckerSaga';
import { getConfigurationBaseUrl } from './deviceConfigurationHelper';
import { areTelemetryIntervalsSame } from './telemetryIntervals';

const debug = Debug('@signageos/front-display:Device:Configuration:deviceConfigurationSagas');

export function* updatingConfiguration(
	getNativeDriver: () => IBasicDriver,
	getManagementDriver: () => IFrontManagementDriver,
	propertyStorage: IPropertyStorage,
	defaultConfig: IConfig,
	publicKey: string,
	threadName: 'front' | 'management',
) {
	let lastConfig: IOptionalConfig<string | null> = {
		updatedAt: null,
		baseUrl: null,
		platformUri: null,
		socketDriver: null,
		staticBaseUrl: null,
		uploadBaseUrl: null,
		weinreUri: null,
		extendedManagementUrl: null,
		subscriptionType: null,
		checkInterval: null,
		telemetryIntervals: null,
		checkTimeBeforeConnection: false,
		offlineActionsLimits: null,
		featureFlags: null,
	};
	yield takeEvery(StartApplication, function* () {
		yield initiateSubscriptionType(propertyStorage, lastConfig, defaultConfig.subscriptionType);
		const deviceUid: string = yield getNativeDriver().getDeviceUid();
		const configBaseUrl: string = yield getConfigurationBaseUrl(getNativeDriver(), defaultConfig);
		const overidenDefaultConfig = {
			...defaultConfig,
			baseUrl: configBaseUrl,
		};
		const isEnabled: boolean = yield getManagementDriver().isDebugEnabled();
		if (isEnabled) {
			// Debug checking is enabled after restart
			overidenDefaultConfig.checkInterval = DEBUG_CHECK_INTERVAL_MS;
		}
		yield* checkDeviceConfigurationSaga(deviceUid, overidenDefaultConfig, publicKey, undefined, threadName);
	});
	yield takeEvery(ConfigurationChanged, function* ({ config }: ConfigurationChanged) {
		debug(threadName, 'new configuration', config);
		if (config.baseUrl && config.baseUrl !== lastConfig.baseUrl) {
			yield put<UpdateBaseUrl>({
				type: UpdateBaseUrl,
				baseUrl: config.baseUrl,
			});
		}
		if (config.staticBaseUrl && config.staticBaseUrl !== lastConfig.staticBaseUrl) {
			yield injectDependencies({
				staticBaseUrl: config.staticBaseUrl,
			});
		}
		if (config.uploadBaseUrl && config.uploadBaseUrl !== lastConfig.uploadBaseUrl) {
			yield injectDependencies({
				uploadBaseUrl: config.uploadBaseUrl,
			});
		}
		if (config.telemetryIntervals && !areTelemetryIntervalsSame(config.telemetryIntervals, lastConfig.telemetryIntervals)) {
			yield injectDependencies({
				telemetryIntervals: config.telemetryIntervals,
			});
		}
		if (config.offlineActionsLimits && config.offlineActionsLimits !== lastConfig.offlineActionsLimits) {
			yield injectDependencies({
				offlineActionsLimits: config.offlineActionsLimits,
			});
		}
		if (config.weinreUri !== lastConfig.weinreUri) {
			yield propertyStorage.setValue(Property.DEBUG_WEINRE_SERVER_URL, config.weinreUri);
		}
		if (config.subscriptionType !== lastConfig.subscriptionType) {
			yield putSubscriptionType(config.subscriptionType || defaultConfig.subscriptionType);
			yield propertyStorage.setValue(Property.INITIAL_SUBSCRIPTION_TYPE, config.subscriptionType);
		}
		if (config.featureFlags && config.featureFlags !== lastConfig.featureFlags) {
			yield injectDependencies({
				featureFlags: config.featureFlags,
			});
		}
		if (
			config.platformUri !== lastConfig.platformUri ||
			config.socketDriver !== lastConfig.socketDriver ||
			config.updatedAt !== lastConfig.updatedAt ||
			config.subscriptionType !== lastConfig.subscriptionType ||
			config.checkTimeBeforeConnection !== lastConfig.checkTimeBeforeConnection
		) {
			yield put({
				type: PlatformConnect,
				platformUri: config.platformUri,
				driver: config.socketDriver,
				checkTimeBeforeConnection: config.checkTimeBeforeConnection,
			} as PlatformConnect);
		}
		lastConfig = config;
	});
}

function* putSubscriptionType(subscriptionType: SubscriptionType) {
	yield injectDependencies({
		subscriptionType: subscriptionType,
	});
	switch (subscriptionType) {
		case SubscriptionType.platform:
			yield put({ type: ChangeSubscriptionToPlatform } as ChangeSubscriptionToPlatform);
			break;
		case SubscriptionType.open:
		default:
			yield put({ type: ChangeSubscriptionToOpen } as ChangeSubscriptionToOpen);
	}
}

/** lastConfig is mutated in key subscriptionType */
function* initiateSubscriptionType(
	propertyStorage: IPropertyStorage,
	lastConfig: IOptionalConfig<string | null>,
	defaultSubscriptionType: SubscriptionType,
) {
	const initialSubscriptionType: SubscriptionType | null = yield propertyStorage.getValueOrDefault(
		Property.INITIAL_SUBSCRIPTION_TYPE,
		null,
	);
	// TODO Remove the artificial delay. Until then:
	// This MUST NOT emit subscription type before all `runPeriodicTaskSagaWhile*` calls get a chance to register callbacks.
	yield wait(500);
	yield putSubscriptionType(initialSubscriptionType || defaultSubscriptionType);
	lastConfig.subscriptionType = initialSubscriptionType;
}
